#!/bin/bash
# ABOUTME: List Linear projects using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
LIMIT=50

show_help() {
    cat << EOF
Usage: $(basename "$0") [OPTIONS]

List Linear projects for the configured team

Optional:
    --limit NUM           Number of projects to return (default: 50)
    --help                Show this help

Examples:
    $(basename "$0")
    $(basename "$0") --limit 10
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Build GraphQL query - query through team to get projects
QUERY='query TeamProjects($teamId: String!, $first: Int) {
  team(id: $teamId) {
    projects(first: $first) {
      nodes {
        id
        name
        description
        state
        priority
        url
        createdAt
        updatedAt
        lead {
          id
          name
          email
        }
      }
    }
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":{\"teamId\":\"$LINEAR_TEAM_ID\",\"first\":$LIMIT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error listing projects:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '.data.team.projects.nodes'
