#!/bin/bash
# ABOUTME: Add a label to a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ISSUE_ID=""
LABEL_ID=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --issue-id ID --label-id ID [OPTIONS]

Add a label to a Linear issue

Required:
    --issue-id ID         Issue ID to add label to
    --label-id ID         Label ID to add

Optional:
    --help                Show this help

Examples:
    $(basename "$0") --issue-id "abc123" --label-id "label456"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --issue-id)
            ISSUE_ID="$2"
            shift 2
            ;;
        --label-id)
            LABEL_ID="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ISSUE_ID" ] || [ -z "$LABEL_ID" ]; then
    echo "Error: --issue-id and --label-id are required"
    show_help
    exit 1
fi

# First get current labels
QUERY='query Issue($id: String!) {
  issue(id: $id) {
    id
    labels {
      nodes {
        id
      }
    }
  }
}'

RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":{\"id\":\"$ISSUE_ID\"}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error getting issue labels:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Extract current label IDs and add new one
CURRENT_LABELS=$(echo "$RESPONSE" | jq -r '.data.issue.labels.nodes[].id')
ALL_LABELS="[\"$LABEL_ID\""
if [ -n "$CURRENT_LABELS" ]; then
    for label in $CURRENT_LABELS; do
        if [ "$label" != "$LABEL_ID" ]; then
            ALL_LABELS="$ALL_LABELS,\"$label\""
        fi
    done
fi
ALL_LABELS="$ALL_LABELS]"

# Build GraphQL mutation
MUTATION='mutation IssueUpdate($id: String!, $input: IssueUpdateInput!) {
  issueUpdate(id: $id, input: $input) {
    success
    issue {
      id
      identifier
      title
      labels {
        nodes {
          id
          name
        }
      }
    }
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"id\":\"$ISSUE_ID\",\"input\":{\"labelIds\":$ALL_LABELS}}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error adding label:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.issueUpdate.success,
  id: .data.issueUpdate.issue.id,
  identifier: .data.issueUpdate.issue.identifier,
  title: .data.issueUpdate.issue.title,
  labels: .data.issueUpdate.issue.labels.nodes
}'
