#!/bin/bash
# ABOUTME: Create an attachment for a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ISSUE_ID=""
URL=""
TITLE=""
SUBTITLE=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --issue-id ID --url URL [OPTIONS]

Create an attachment for a Linear issue (link external resources)

Required:
    --issue-id ID         Issue ID to attach to
    --url URL             URL of the external resource

Optional:
    --title TEXT          Attachment title
    --subtitle TEXT       Attachment subtitle/description
    --help                Show this help

Examples:
    $(basename "$0") --issue-id "abc123" --url "https://github.com/org/repo/pull/123"
    $(basename "$0") --issue-id "abc123" --url "https://figma.com/..." --title "Design mockup"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --issue-id)
            ISSUE_ID="$2"
            shift 2
            ;;
        --url)
            URL="$2"
            shift 2
            ;;
        --title)
            TITLE="$2"
            shift 2
            ;;
        --subtitle)
            SUBTITLE="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ISSUE_ID" ] || [ -z "$URL" ]; then
    echo "Error: --issue-id and --url are required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation AttachmentCreate($input: AttachmentCreateInput!) {
  attachmentCreate(input: $input) {
    success
    attachment {
      id
      title
      subtitle
      url
      createdAt
    }
  }
}'

# Build input object
INPUT="{\"issueId\":\"$ISSUE_ID\",\"url\":\"$URL\""

if [ -n "$TITLE" ]; then
    TITLE_ESCAPED=$(echo "$TITLE" | jq -Rs .)
    INPUT="$INPUT,\"title\":$TITLE_ESCAPED"
fi

if [ -n "$SUBTITLE" ]; then
    SUBTITLE_ESCAPED=$(echo "$SUBTITLE" | jq -Rs .)
    INPUT="$INPUT,\"subtitle\":$SUBTITLE_ESCAPED"
fi

INPUT="$INPUT}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating attachment:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.attachmentCreate.success,
  id: .data.attachmentCreate.attachment.id,
  title: .data.attachmentCreate.attachment.title,
  subtitle: .data.attachmentCreate.attachment.subtitle,
  url: .data.attachmentCreate.attachment.url,
  createdAt: .data.attachmentCreate.attachment.createdAt
}'
