#!/bin/bash
# ABOUTME: Create a comment on a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ISSUE_ID=""
BODY=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --issue-id ID --body "Comment text" [OPTIONS]

Create a comment on a Linear issue

Required:
    --issue-id ID         Issue ID to comment on
    --body TEXT           Comment body (markdown supported)

Optional:
    --help                Show this help

Examples:
    $(basename "$0") --issue-id "abc123" --body "Work in progress"
    $(basename "$0") --issue-id "abc123" --body "Fixed in PR #123"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --issue-id)
            ISSUE_ID="$2"
            shift 2
            ;;
        --body)
            BODY="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ISSUE_ID" ] || [ -z "$BODY" ]; then
    echo "Error: --issue-id and --body are required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation CommentCreate($input: CommentCreateInput!) {
  commentCreate(input: $input) {
    success
    comment {
      id
      body
      createdAt
      updatedAt
      user {
        id
        name
        email
      }
      issue {
        id
        identifier
        title
      }
    }
  }
}'

# Escape body for JSON
BODY_ESCAPED=$(echo "$BODY" | jq -Rs .)

# Build input object
INPUT="{\"issueId\":\"$ISSUE_ID\",\"body\":$BODY_ESCAPED}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating comment:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.commentCreate.success,
  id: .data.commentCreate.comment.id,
  body: .data.commentCreate.comment.body,
  createdAt: .data.commentCreate.comment.createdAt,
  user: .data.commentCreate.comment.user,
  issue: .data.commentCreate.comment.issue
}'
