#!/bin/bash
# ABOUTME: Create a cycle for a team using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
NAME=""
DESCRIPTION=""
STARTS_AT=""
ENDS_AT=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --starts-at DATE --ends-at DATE [OPTIONS]

Create a cycle for the configured team

Required:
    --starts-at DATE      Cycle start date (ISO 8601 format: YYYY-MM-DD)
    --ends-at DATE        Cycle end date (ISO 8601 format: YYYY-MM-DD)

Optional:
    --name TEXT           Cycle name (auto-generated if not provided)
    --description TEXT    Cycle description
    --help                Show this help

Examples:
    $(basename "$0") --starts-at "2025-11-01" --ends-at "2025-11-14"
    $(basename "$0") --name "Sprint 42" --description "Q4 goals" --starts-at "2025-11-01" --ends-at "2025-11-14"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --name)
            NAME="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --starts-at)
            STARTS_AT="$2"
            shift 2
            ;;
        --ends-at)
            ENDS_AT="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$STARTS_AT" ] || [ -z "$ENDS_AT" ]; then
    echo "Error: --starts-at and --ends-at are required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation CycleCreate($input: CycleCreateInput!) {
  cycleCreate(input: $input) {
    success
    cycle {
      id
      name
      number
      description
      startsAt
      endsAt
    }
  }
}'

# Build input object
INPUT="{\"teamId\":\"$LINEAR_TEAM_ID\",\"startsAt\":\"$STARTS_AT\",\"endsAt\":\"$ENDS_AT\""

if [ -n "$NAME" ]; then
    NAME_ESCAPED=$(echo "$NAME" | jq -Rs .)
    INPUT="$INPUT,\"name\":$NAME_ESCAPED"
fi

if [ -n "$DESCRIPTION" ]; then
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    INPUT="$INPUT,\"description\":$DESC_ESCAPED"
fi

INPUT="$INPUT}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating cycle:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.cycleCreate.success,
  id: .data.cycleCreate.cycle.id,
  name: .data.cycleCreate.cycle.name,
  number: .data.cycleCreate.cycle.number,
  startsAt: .data.cycleCreate.cycle.startsAt,
  endsAt: .data.cycleCreate.cycle.endsAt
}'
