#!/bin/bash
# ABOUTME: Create a Linear project using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
NAME=""
DESCRIPTION=""
COLOR=""
TARGET_DATE=""
LEAD_ID=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --name "Project Name" [OPTIONS]

Create a Linear project

Required:
    --name TEXT           Project name

Optional:
    --description TEXT    Project description
    --color TEXT          Project color (hex code)
    --target-date DATE    Target completion date
    --lead-id ID          Project lead user ID
    --help                Show this help

Examples:
    $(basename "$0") --name "Backend Framework"
    $(basename "$0") --name "UI Refactor" --description "Component updates" --color "#FF5733"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --name)
            NAME="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --color)
            COLOR="$2"
            shift 2
            ;;
        --target-date)
            TARGET_DATE="$2"
            shift 2
            ;;
        --lead-id)
            LEAD_ID="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$NAME" ]; then
    echo "Error: --name is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation ProjectCreate($input: ProjectCreateInput!) {
  projectCreate(input: $input) {
    success
    project {
      id
      name
      description
      url
      state
    }
  }
}'

# Build input object
INPUT="{\"teamIds\":[\"$LINEAR_TEAM_ID\"],\"name\":\"$NAME\""

if [ -n "$DESCRIPTION" ]; then
    # Escape description for JSON
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    INPUT="$INPUT,\"description\":$DESC_ESCAPED"
fi

if [ -n "$COLOR" ]; then
    INPUT="$INPUT,\"color\":\"$COLOR\""
fi

if [ -n "$TARGET_DATE" ]; then
    INPUT="$INPUT,\"targetDate\":\"$TARGET_DATE\""
fi

if [ -n "$LEAD_ID" ]; then
    INPUT="$INPUT,\"leadId\":\"$LEAD_ID\""
fi

INPUT="$INPUT}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating project:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.projectCreate.success,
  id: .data.projectCreate.project.id,
  name: .data.projectCreate.project.name,
  description: .data.projectCreate.project.description,
  url: .data.projectCreate.project.url,
  state: .data.projectCreate.project.state
}'
