#!/bin/bash
# ABOUTME: Delete an attachment from a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ATTACHMENT_ID=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --attachment-id ID [OPTIONS]

Delete an attachment from a Linear issue

Required:
    --attachment-id ID    Attachment ID to delete

Optional:
    --help                Show this help

Examples:
    $(basename "$0") --attachment-id "abc123"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --attachment-id)
            ATTACHMENT_ID="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ATTACHMENT_ID" ]; then
    echo "Error: --attachment-id is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation AttachmentDelete($id: String!) {
  attachmentDelete(id: $id) {
    success
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"id\":\"$ATTACHMENT_ID\"}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error deleting attachment:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.attachmentDelete.success
}'
