#!/bin/bash
# ABOUTME: List cycles for a team using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
LIMIT=20
INCLUDE_COMPLETED="false"

show_help() {
    cat << EOF
Usage: $(basename "$0") [OPTIONS]

List cycles for the configured team

Optional:
    --limit NUM              Number of cycles to return (default: 20)
    --include-completed      Include completed cycles
    --help                   Show this help

Examples:
    $(basename "$0")
    $(basename "$0") --include-completed --limit 10
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --include-completed)
            INCLUDE_COMPLETED="true"
            shift
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Build GraphQL query
QUERY='query TeamCycles($teamId: String!, $first: Int, $filter: CycleFilter) {
  team(id: $teamId) {
    cycles(first: $first, filter: $filter) {
      nodes {
        id
        name
        number
        startsAt
        endsAt
        completedAt
        description
        progress
        issues {
          nodes {
            id
            identifier
            title
          }
        }
      }
    }
  }
}'

# Build filter
FILTER="{}"
if [ "$INCLUDE_COMPLETED" = "false" ]; then
    FILTER='{"isCompleted":{"eq":false}}'
fi

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":{\"teamId\":\"$LINEAR_TEAM_ID\",\"first\":$LIMIT,\"filter\":$FILTER}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error listing cycles:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '.data.team.cycles.nodes'
