#!/bin/bash
# ABOUTME: List workflow states for a team using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
LIMIT=50

show_help() {
    cat << EOF
Usage: $(basename "$0") [OPTIONS]

List workflow states for the configured team

Optional:
    --limit NUM           Number of states to return (default: 50)
    --help                Show this help

Examples:
    $(basename "$0")
    $(basename "$0") --limit 20
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Build GraphQL query
QUERY='query TeamStates($teamId: String!, $first: Int) {
  team(id: $teamId) {
    states(first: $first) {
      nodes {
        id
        name
        description
        type
        color
        position
      }
    }
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":{\"teamId\":\"$LINEAR_TEAM_ID\",\"first\":$LIMIT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error listing states:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '.data.team.states.nodes'
