#!/bin/bash
# ABOUTME: Search Linear issues using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
QUERY=""
LIMIT=20
AFTER=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --query "search terms" [OPTIONS]

Search Linear issues by query string with pagination

Required:
    --query TEXT          Search query

Optional:
    --limit NUM           Number of results to return (default: 20)
    --after CURSOR        Pagination cursor for next page
    --help                Show this help

Examples:
    $(basename "$0") --query "authentication"
    $(basename "$0") --query "bug fix" --limit 10
    $(basename "$0") --query "feature" --after "cursor123"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --query)
            QUERY="$2"
            shift 2
            ;;
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --after)
            AFTER="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$QUERY" ]; then
    echo "Error: --query is required"
    show_help
    exit 1
fi

# Build GraphQL query
GQL_QUERY='query IssueSearch($query: String!, $filter: IssueFilter, $first: Int, $after: String) {
  issueSearch(query: $query, filter: $filter, first: $first, after: $after) {
    nodes {
      id
      identifier
      title
      description
      priority
      url
      createdAt
      updatedAt
      assignee {
        id
        name
        email
      }
      state {
        id
        name
        type
      }
      project {
        id
        name
      }
    }
    pageInfo {
      hasNextPage
      endCursor
    }
  }
}'

# Build filter for team
FILTER="{\"team\":{\"id\":{\"eq\":\"$LINEAR_TEAM_ID\"}}}"

# Build variables object
VARIABLES="{\"query\":\"$QUERY\",\"filter\":$FILTER,\"first\":$LIMIT"
if [ -n "$AFTER" ]; then
    VARIABLES="$VARIABLES,\"after\":\"$AFTER\""
fi
VARIABLES="$VARIABLES}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$GQL_QUERY" | jq -Rs .),\"variables\":$VARIABLES}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error searching issues:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result with pagination info
echo "$RESPONSE" | jq '{
  issues: .data.issueSearch.nodes,
  pageInfo: .data.issueSearch.pageInfo
}'
