#!/bin/bash
# ABOUTME: Update a comment on a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
COMMENT_ID=""
BODY=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --comment-id ID --body "Updated text" [OPTIONS]

Update a comment on a Linear issue

Required:
    --comment-id ID       Comment ID to update
    --body TEXT           New comment body (markdown supported)

Optional:
    --help                Show this help

Examples:
    $(basename "$0") --comment-id "abc123" --body "Updated status"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --comment-id)
            COMMENT_ID="$2"
            shift 2
            ;;
        --body)
            BODY="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$COMMENT_ID" ] || [ -z "$BODY" ]; then
    echo "Error: --comment-id and --body are required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation CommentUpdate($id: String!, $input: CommentUpdateInput!) {
  commentUpdate(id: $id, input: $input) {
    success
    comment {
      id
      body
      updatedAt
    }
  }
}'

# Escape body for JSON
BODY_ESCAPED=$(echo "$BODY" | jq -Rs .)

# Build input object
INPUT="{\"body\":$BODY_ESCAPED}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"id\":\"$COMMENT_ID\",\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error updating comment:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.commentUpdate.success,
  id: .data.commentUpdate.comment.id,
  body: .data.commentUpdate.comment.body,
  updatedAt: .data.commentUpdate.comment.updatedAt
}'
