#!/bin/bash
# ABOUTME: Update a cycle using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
CYCLE_ID=""
NAME=""
DESCRIPTION=""
STARTS_AT=""
ENDS_AT=""
COMPLETED_AT=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --cycle-id ID [OPTIONS]

Update a cycle

Required:
    --cycle-id ID         Cycle ID to update

Optional:
    --name TEXT           New cycle name
    --description TEXT    New cycle description
    --starts-at DATE      New start date (ISO 8601 format: YYYY-MM-DD)
    --ends-at DATE        New end date (ISO 8601 format: YYYY-MM-DD)
    --completed-at DATE   Mark cycle completed (ISO 8601 format, use "null" to unmark)
    --help                Show this help

Examples:
    $(basename "$0") --cycle-id "abc123" --name "Sprint 43"
    $(basename "$0") --cycle-id "abc123" --completed-at "2025-11-14"
    $(basename "$0") --cycle-id "abc123" --completed-at "null"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --cycle-id)
            CYCLE_ID="$2"
            shift 2
            ;;
        --name)
            NAME="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --starts-at)
            STARTS_AT="$2"
            shift 2
            ;;
        --ends-at)
            ENDS_AT="$2"
            shift 2
            ;;
        --completed-at)
            COMPLETED_AT="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$CYCLE_ID" ]; then
    echo "Error: --cycle-id is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation CycleUpdate($id: String!, $input: CycleUpdateInput!) {
  cycleUpdate(id: $id, input: $input) {
    success
    cycle {
      id
      name
      number
      description
      startsAt
      endsAt
      completedAt
    }
  }
}'

# Build input object
INPUT="{"
FIELDS=()

if [ -n "$NAME" ]; then
    NAME_ESCAPED=$(echo "$NAME" | jq -Rs .)
    FIELDS+=("\"name\":$NAME_ESCAPED")
fi

if [ -n "$DESCRIPTION" ]; then
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    FIELDS+=("\"description\":$DESC_ESCAPED")
fi

if [ -n "$STARTS_AT" ]; then
    FIELDS+=("\"startsAt\":\"$STARTS_AT\"")
fi

if [ -n "$ENDS_AT" ]; then
    FIELDS+=("\"endsAt\":\"$ENDS_AT\"")
fi

if [ -n "$COMPLETED_AT" ]; then
    if [ "$COMPLETED_AT" = "null" ]; then
        FIELDS+=("\"completedAt\":null")
    else
        FIELDS+=("\"completedAt\":\"$COMPLETED_AT\"")
    fi
fi

# Join fields with commas
INPUT="$INPUT$(IFS=,; echo "${FIELDS[*]}")}"

if [ ${#FIELDS[@]} -eq 0 ]; then
    echo "Error: At least one field to update must be specified"
    show_help
    exit 1
fi

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"id\":\"$CYCLE_ID\",\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error updating cycle:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.cycleUpdate.success,
  id: .data.cycleUpdate.cycle.id,
  name: .data.cycleUpdate.cycle.name,
  number: .data.cycleUpdate.cycle.number,
  startsAt: .data.cycleUpdate.cycle.startsAt,
  endsAt: .data.cycleUpdate.cycle.endsAt,
  completedAt: .data.cycleUpdate.cycle.completedAt
}'
