#!/bin/bash
# ABOUTME: Update a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ISSUE_ID=""
TITLE=""
DESCRIPTION=""
STATE_ID=""
ASSIGNEE_ID=""
PRIORITY=""
PROJECT_ID=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --issue-id ID [OPTIONS]

Update a Linear issue

Required:
    --issue-id ID         Issue ID to update

Optional:
    --title TEXT          New title
    --description TEXT    New description
    --state-id ID         New workflow state ID
    --assignee-id ID      New assignee user ID (use "null" to unassign)
    --priority NUM        New priority (0=none, 1=urgent, 2=high, 3=medium, 4=low)
    --project-id ID       New project ID (use "null" to remove)
    --help                Show this help

Examples:
    $(basename "$0") --issue-id "abc123" --title "Updated title"
    $(basename "$0") --issue-id "abc123" --state-id "state456" --priority 1
    $(basename "$0") --issue-id "abc123" --assignee-id "null"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --issue-id)
            ISSUE_ID="$2"
            shift 2
            ;;
        --title)
            TITLE="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --state-id)
            STATE_ID="$2"
            shift 2
            ;;
        --assignee-id)
            ASSIGNEE_ID="$2"
            shift 2
            ;;
        --priority)
            PRIORITY="$2"
            shift 2
            ;;
        --project-id)
            PROJECT_ID="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ISSUE_ID" ]; then
    echo "Error: --issue-id is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation IssueUpdate($id: String!, $input: IssueUpdateInput!) {
  issueUpdate(id: $id, input: $input) {
    success
    issue {
      id
      identifier
      title
      url
    }
  }
}'

# Build input object
INPUT="{"

FIELDS=()

if [ -n "$TITLE" ]; then
    TITLE_ESCAPED=$(echo "$TITLE" | jq -Rs .)
    FIELDS+=("\"title\":$TITLE_ESCAPED")
fi

if [ -n "$DESCRIPTION" ]; then
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    FIELDS+=("\"description\":$DESC_ESCAPED")
fi

if [ -n "$STATE_ID" ]; then
    FIELDS+=("\"stateId\":\"$STATE_ID\"")
fi

if [ -n "$ASSIGNEE_ID" ]; then
    if [ "$ASSIGNEE_ID" = "null" ]; then
        FIELDS+=("\"assigneeId\":null")
    else
        FIELDS+=("\"assigneeId\":\"$ASSIGNEE_ID\"")
    fi
fi

if [ -n "$PRIORITY" ]; then
    FIELDS+=("\"priority\":$PRIORITY")
fi

if [ -n "$PROJECT_ID" ]; then
    if [ "$PROJECT_ID" = "null" ]; then
        FIELDS+=("\"projectId\":null")
    else
        FIELDS+=("\"projectId\":\"$PROJECT_ID\"")
    fi
fi

# Join fields with commas
INPUT="$INPUT$(IFS=,; echo "${FIELDS[*]}")}"

if [ ${#FIELDS[@]} -eq 0 ]; then
    echo "Error: At least one field to update must be specified"
    show_help
    exit 1
fi

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"id\":\"$ISSUE_ID\",\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error updating issue:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.issueUpdate.success,
  id: .data.issueUpdate.issue.id,
  identifier: .data.issueUpdate.issue.identifier,
  title: .data.issueUpdate.issue.title,
  url: .data.issueUpdate.issue.url
}'
