# Tech Debt Detection Patterns

Regex patterns and heuristics for code analysis.

## Commented-Out Code Detection

### TypeScript/JavaScript

```bash
# Multi-line commented code blocks (likely old code)
rg -U '^\s*/\*[\s\S]*?(function|const|let|var|class|import|export)[\s\S]*?\*/' --type ts

# Single-line commented code (looks like statements)
rg '^\s*//\s*(const|let|var|function|class|return|if|for|while)\s+\w+' --type ts

# Commented imports
rg '^\s*//\s*import\s+' --type ts
```

### Python

```bash
# Commented function definitions
rg '^\s*#\s*def\s+\w+\(' --type py

# Commented class definitions
rg '^\s*#\s*class\s+\w+' --type py

# Multi-line string blocks that look like code
rg -U '"""[\s\S]*?(def|class|import|return)[\s\S]*?"""' --type py
```

## Unreachable Code Patterns

### After Return/Throw

```bash
# TypeScript/JavaScript - code after return
rg -U 'return\s+[^;]+;\s*\n\s*[a-zA-Z]' --type ts

# Code after throw
rg -U 'throw\s+[^;]+;\s*\n\s*[a-zA-Z]' --type ts

# Python - code after return (same indentation)
rg -U 'return\s+.*\n(\s+)\S' --type py
```

### Dead Branches

```bash
# if (false) patterns
rg 'if\s*\(\s*false\s*\)' --type ts

# if (true) with else (dead else)
rg -U 'if\s*\(\s*true\s*\)\s*\{[^}]*\}\s*else' --type ts

# Python: if False:
rg 'if\s+False\s*:' --type py
```

## Unused Variable Patterns

### Declaration Without Use

```bash
# TypeScript - const declared, search for no other refs
# (Requires cross-referencing, use eslint instead)

# Python - assigned but starts with _
rg '^\s*_\w+\s*=' --type py
```

## Version/Backup File Patterns

Files that are likely old versions:

```bash
# Common version suffixes
find . -type f \( \
  -name "*-old.*" -o \
  -name "*-backup.*" -o \
  -name "*-copy.*" -o \
  -name "*-new.*" -o \
  -name "*-fixed.*" -o \
  -name "*-v[0-9]*.*" -o \
  -name "*.bak" -o \
  -name "*.backup" -o \
  -name "*.orig" \
\) -not -path "*/node_modules/*" -not -path "*/.git/*"

# Common version prefixes
find . -type f \( \
  -name "old-*" -o \
  -name "backup-*" -o \
  -name "copy-*" -o \
  -name "temp-*" -o \
  -name "tmp-*" \
\) -not -path "*/node_modules/*" -not -path "*/.git/*"
```

## Empty/Stub File Detection

### TypeScript/JavaScript

```bash
# Files with only imports (no exports or code)
for f in $(find . -name "*.ts" -not -path "*/node_modules/*"); do
  exports=$(grep -c 'export' "$f" || echo 0)
  lines=$(wc -l < "$f" | tr -d ' ')
  if [[ $exports -eq 0 && $lines -lt 10 ]]; then
    echo "Potentially empty: $f"
  fi
done

# Files that only re-export
rg -l '^export \* from' --type ts | while read f; do
  other_content=$(grep -v '^export \* from\|^import\|^//' "$f" | grep -v '^\s*$' | wc -l)
  if [[ $other_content -eq 0 ]]; then
    echo "Re-export only: $f"
  fi
done
```

### Python

```bash
# Empty __init__.py (often intentional, but check)
find . -name "__init__.py" -empty -not -path "*/.venv/*"

# Files with only pass
for f in $(find . -name "*.py" -not -path "*/.venv/*"); do
  content=$(grep -v '^\s*#\|^\s*$\|^\s*pass\s*$\|^"""' "$f" | wc -l)
  if [[ $content -lt 3 ]]; then
    echo "Potentially stub: $f"
  fi
done
```

## Duplicate Code Heuristics

### Function Signature Similarity

```bash
# Find functions with same name different files
rg -o 'function\s+(\w+)\s*\(' --type ts -r '$1' | \
  sort | uniq -d | while read func; do
    echo "Duplicate function name: $func"
    rg -l "function\s+$func\s*\(" --type ts
  done

# Python
rg -o 'def\s+(\w+)\s*\(' --type py -r '$1' | \
  sort | uniq -d | while read func; do
    echo "Duplicate function name: $func"
    rg -l "def\s+$func\s*\(" --type py
  done
```

### Import Pattern Clustering

Files with nearly identical imports often have duplicated code:

```bash
# Extract and hash import blocks
for f in $(find . -name "*.ts" -not -path "*/node_modules/*"); do
  imports=$(grep '^import' "$f" | sort | md5)
  echo "$imports $f"
done | sort | uniq -D -w 32
```

## Complexity Indicators

### Deep Nesting

```bash
# TypeScript/JavaScript - lines with 5+ levels of indentation
rg '^\s{20,}' --type ts

# Python - 5+ levels (assuming 4-space indent)
rg '^\s{20,}' --type py
```

### Long Functions

```bash
# Find function boundaries and count lines
# (Complex - better to use radon/eslint-complexity)

# Quick heuristic: functions with many lines between braces
awk '/function\s+\w+.*\{/{start=NR} /^\s*\}/{if(NR-start>50)print FILENAME":"start" ("NR-start" lines)"}' *.ts
```

### Many Parameters

```bash
# Functions with >5 parameters
rg 'function\s+\w+\s*\([^)]{80,}\)' --type ts
rg 'def\s+\w+\s*\([^)]{80,}\)' --type py
```

## TODO/FIXME Analysis

### Count by Type

```bash
# Categorize TODOs
echo "=== TODO ===" && rg -c 'TODO' --type-add 'code:*.{ts,js,py}' -t code | sort -t: -k2 -nr | head -10
echo "=== FIXME ===" && rg -c 'FIXME' --type-add 'code:*.{ts,js,py}' -t code | sort -t: -k2 -nr | head -10
echo "=== HACK ===" && rg -c 'HACK' --type-add 'code:*.{ts,js,py}' -t code | sort -t: -k2 -nr | head -10
echo "=== XXX ===" && rg -c 'XXX' --type-add 'code:*.{ts,js,py}' -t code | sort -t: -k2 -nr | head -10
```

### Age Analysis

```bash
# TODOs in old files (not modified in 90 days)
find . -name "*.ts" -mtime +90 -not -path "*/node_modules/*" | \
  xargs rg 'TODO|FIXME' 2>/dev/null
```

### Orphaned TODOs

TODOs referencing issues/tickets that may be closed:

```bash
# Find issue references
rg 'TODO.*#[0-9]+|FIXME.*#[0-9]+' --type-add 'code:*.{ts,js,py}' -t code
```

## Deprecated API Patterns

### Node.js

```bash
# Deprecated fs methods
rg 'fs\.(exists|existsSync)\(' --type ts

# Old callback patterns where promises available
rg 'fs\.(readFile|writeFile)\([^)]+,\s*function' --type ts
```

### React

```bash
# Deprecated lifecycle methods
rg 'componentWillMount|componentWillReceiveProps|componentWillUpdate' --type tsx

# Old context API
rg 'contextTypes|childContextTypes|getChildContext' --type tsx
```

### Python

```bash
# Python 2 patterns in Python 3 code
rg 'print\s+["\']|print\s+\(' --type py  # print statement vs function
rg 'from __future__ import' --type py    # May indicate legacy code
rg '\.iteritems\(\)|\.itervalues\(\)|\.iterkeys\(\)' --type py
```

## False Positive Filters

### Intentionally Unused

```bash
# Underscore prefix (intentionally unused)
# Skip these in unused variable detection
rg '^\s*(const|let|var)\s+_\w+' --type ts

# Python underscore convention
rg '^\s*_\w+\s*=' --type py
```

### Test Mocks

```bash
# Mock/stub patterns - may appear unused but are test fixtures
rg 'mock\w*|stub\w*|fake\w*' -l --type ts
```

### Public API

```bash
# index.ts re-exports are intentionally for external use
# Don't flag as unused exports
rg '^export \* from|^export \{' -l --type ts | grep 'index.ts'
```

## Confidence Scoring

### High Confidence (Tier 1) Indicators

- Tool reports with 100% confidence
- Pattern matches known safe patterns (backup files, etc.)
- Multiple tools agree on same issue
- Zero references found across entire codebase

### Medium Confidence (Tier 2) Indicators

- Tool reports 80-99% confidence
- Single tool detection
- Some references exist but appear to be in tests only
- File hasn't been modified in >90 days

### Low Confidence (Tier 3) Indicators

- Tool reports <80% confidence
- Dynamic usage possible (reflection, eval, etc.)
- Public API (might be used by external consumers)
- Recently modified file
