# Tech Debt Detection Tool Commands

Detailed commands for each language stack.

## TypeScript/JavaScript

### Knip (Recommended - Comprehensive)

Knip finds unused files, dependencies, and exports in one tool.

```bash
# Install
npm install -D knip

# Run full analysis
npx knip

# JSON output for parsing
npx knip --reporter json

# Specific checks
npx knip --include files        # Unused files only
npx knip --include dependencies # Unused deps only
npx knip --include exports      # Unused exports only
```

**Parse output:**
```bash
npx knip --reporter json | jq '.files[]' # List unused files
npx knip --reporter json | jq '.exports[]' # List unused exports
```

### ts-prune (TypeScript Exports)

Finds exports that are never imported.

```bash
# Install
npm install -D ts-prune

# Run
npx ts-prune

# Ignore index files
npx ts-prune --ignore 'index.ts'

# JSON output
npx ts-prune | grep -v '(used in module)'
```

**Output format:**
```
src/utils/helper.ts:15 - unusedFunction
src/types/old.ts:3 - OldInterface
```

### ESLint Dead Code Rules

```bash
# Check unused vars
npx eslint . --rule 'no-unused-vars: error' --format json

# Check unreachable code
npx eslint . --rule 'no-unreachable: error' --format json

# Combined
npx eslint . --rule '{
  "no-unused-vars": "error",
  "no-unreachable": "error",
  "no-unused-expressions": "error"
}' --format json
```

### Duplicate Detection (jscpd)

```bash
# Install
npm install -g jscpd

# Run
jscpd ./src --min-lines 20 --reporters json

# Output to file
jscpd ./src --min-lines 20 --output ./jscpd-report
```

### Complexity Analysis (eslint-plugin-complexity)

```bash
# In eslint config
{
  "rules": {
    "complexity": ["warn", 15]
  }
}

# Or standalone
npx eslint . --rule 'complexity: ["error", 15]' --format json
```

## Python

### Vulture (Dead Code)

Primary tool for Python dead code detection.

```bash
# Install
pip install vulture

# Basic scan
vulture .

# With confidence threshold (80% = fewer false positives)
vulture . --min-confidence 80

# Generate whitelist for intentional unused code
vulture . --make-whitelist > whitelist.py

# Use whitelist
vulture . whitelist.py
```

**Output format:**
```
src/utils.py:15: unused function 'old_helper' (60% confidence)
src/models.py:42: unused variable 'temp' (100% confidence)
```

### Flake8 (Unused Imports/Vars)

```bash
# Install
pip install flake8

# Check unused imports only
flake8 --select=F401 .

# Check unused variables only
flake8 --select=F841 .

# Combined with JSON output
flake8 --select=F401,F841 --format=json .
```

**Error codes:**
- F401: Module imported but unused
- F841: Local variable assigned but never used

### Pylint (Comprehensive)

```bash
# Install
pip install pylint

# Unused code only
pylint --disable=all --enable=W0611,W0612,W0613 .

# JSON output
pylint --output-format=json --disable=all --enable=W0611,W0612,W0613 .
```

**Message codes:**
- W0611: Unused import
- W0612: Unused variable
- W0613: Unused argument

### Radon (Complexity)

```bash
# Install
pip install radon

# Cyclomatic complexity
radon cc . -a -s --json

# Maintainability index
radon mi . -s --json

# Show only complex functions (C grade or worse)
radon cc . -a -nc
```

**Grades:** A (1-5), B (6-10), C (11-20), D (21-30), E (31-40), F (41+)

### Duplicate Detection (pylint)

```bash
# Duplicate code detection
pylint --disable=all --enable=duplicate-code .

# Or use CPD (Copy-Paste Detector)
pmd cpd --minimum-tokens 50 --language python --files .
```

## Rust

### cargo-udeps (Unused Dependencies)

```bash
# Install (requires nightly)
cargo install cargo-udeps --locked

# Run
cargo +nightly udeps

# JSON output
cargo +nightly udeps --output json
```

### Clippy (Dead Code)

```bash
# Warn on dead code
cargo clippy -- -W dead_code

# Deny (fail on) dead code
cargo clippy -- -D dead_code

# JSON output
cargo clippy --message-format=json -- -W dead_code
```

### Unused Features

```bash
# Check for unused feature flags
cargo clippy -- -W unused_features
```

## Go

### staticcheck (Comprehensive)

```bash
# Install
go install honnef.co/go/tools/cmd/staticcheck@latest

# Run
staticcheck ./...

# Check specific issues
staticcheck -checks U1000 ./...  # Unused code
```

**Codes:**
- U1000: Unused code (funcs, types, vars)

### deadcode

```bash
# Install
go install golang.org/x/tools/cmd/deadcode@latest

# Run
deadcode ./...
```

### go vet

```bash
# Unused variables
go vet ./...
```

### golangci-lint (All-in-one)

```bash
# Install
go install github.com/golangci/golangci-lint/cmd/golangci-lint@latest

# Run unused checks
golangci-lint run --enable=unused,deadcode

# JSON output
golangci-lint run --enable=unused,deadcode --out-format=json
```

## Cross-Language Duplicate Detection

### PMD CPD (Copy-Paste Detector)

Works with: Java, JavaScript, Python, Go, and more.

```bash
# Install via brew
brew install pmd

# Run
pmd cpd --minimum-tokens 50 --language javascript --files ./src

# Multiple languages
pmd cpd --minimum-tokens 50 --files ./src --language-property \
  'minimumTokens=50,language=javascript,python,typescript'
```

### jscpd (JavaScript-based, multi-language)

```bash
# Install
npm install -g jscpd

# Run on mixed codebase
jscpd . --min-lines 15 --reporters json,console

# Ignore patterns
jscpd . --ignore '**/node_modules/**,**/*.test.*'
```

## TODO/FIXME Detection

### ripgrep (Cross-language)

```bash
# Find all TODOs
rg -n 'TODO|FIXME|XXX|HACK' --type-add 'code:*.{ts,js,py,rs,go}' -t code

# Count per file
rg -c 'TODO|FIXME' --type-add 'code:*.{ts,js,py,rs,go}' -t code | sort -t: -k2 -nr

# Files with >5 TODOs
rg -c 'TODO|FIXME' --type-add 'code:*.{ts,js,py,rs,go}' -t code | \
  awk -F: '$2 > 5 {print}'
```

## Complexity Thresholds

| Metric | Acceptable | Warning | Critical |
|--------|-----------|---------|----------|
| Cyclomatic complexity | 1-10 | 11-20 | >20 |
| Function length (lines) | 1-50 | 51-100 | >100 |
| File length (lines) | 1-300 | 301-500 | >500 |
| Nesting depth | 1-3 | 4-5 | >5 |
| Parameters | 1-4 | 5-6 | >6 |

## Output Parsing Patterns

### Common JSON structure for metadata

```bash
# Parse knip JSON
npx knip --reporter json | jq -r '.files[] | {
  path: .path,
  reason: "unused_file",
  tier: 2
}'

# Parse vulture output
vulture . --min-confidence 80 2>&1 | \
  grep -E '^[^:]+:[0-9]+:' | \
  sed 's/:/ /g' | \
  awk '{print "{\"path\": \"" $1 "\", \"line\": " $2 ", \"reason\": \"" $5 "\"}"}'
```
