#!/usr/bin/env bash
# ABOUTME: Install tech debt detection tools for all supported languages
# ABOUTME: Run once per machine to set up tooling

set -e

echo "🔧 Installing Tech Debt Detection Tools"
echo "========================================"

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

installed() {
    echo -e "${GREEN}✓${NC} $1 already installed"
}

installing() {
    echo -e "${YELLOW}→${NC} Installing $1..."
}

# =============================================================================
# JavaScript/TypeScript
# =============================================================================
echo ""
echo "📦 JavaScript/TypeScript Tools"
echo "------------------------------"

# knip - comprehensive unused detection
if command -v npx &> /dev/null && npm list -g knip &> /dev/null; then
    installed "knip"
else
    installing "knip"
    npm install -g knip
fi

# ts-prune - unused exports
if npm list -g ts-prune &> /dev/null 2>&1; then
    installed "ts-prune"
else
    installing "ts-prune"
    npm install -g ts-prune
fi

# jscpd - duplicate detection
if command -v jscpd &> /dev/null; then
    installed "jscpd"
else
    installing "jscpd"
    npm install -g jscpd
fi

# =============================================================================
# Python
# =============================================================================
echo ""
echo "🐍 Python Tools"
echo "---------------"

# vulture - dead code detection
if command -v vulture &> /dev/null; then
    installed "vulture"
else
    installing "vulture"
    pip install vulture
fi

# radon - complexity analysis
if command -v radon &> /dev/null; then
    installed "radon"
else
    installing "radon"
    pip install radon
fi

# flake8 (usually already installed)
if command -v flake8 &> /dev/null; then
    installed "flake8"
else
    installing "flake8"
    pip install flake8
fi

# pylint (usually already installed)
if command -v pylint &> /dev/null; then
    installed "pylint"
else
    installing "pylint"
    pip install pylint
fi

# =============================================================================
# Rust
# =============================================================================
echo ""
echo "🦀 Rust Tools"
echo "-------------"

if command -v cargo &> /dev/null; then
    # cargo-udeps
    if cargo install --list | grep -q "cargo-udeps"; then
        installed "cargo-udeps"
    else
        installing "cargo-udeps"
        cargo install cargo-udeps --locked
    fi
else
    echo "  Rust not installed, skipping Rust tools"
fi

# =============================================================================
# Go
# =============================================================================
echo ""
echo "🐹 Go Tools"
echo "-----------"

if command -v go &> /dev/null; then
    # staticcheck
    if command -v staticcheck &> /dev/null; then
        installed "staticcheck"
    else
        installing "staticcheck"
        go install honnef.co/go/tools/cmd/staticcheck@latest
    fi

    # deadcode
    if command -v deadcode &> /dev/null; then
        installed "deadcode"
    else
        installing "deadcode"
        go install golang.org/x/tools/cmd/deadcode@latest
    fi
else
    echo "  Go not installed, skipping Go tools"
fi

# =============================================================================
# Cross-language
# =============================================================================
echo ""
echo "🌐 Cross-language Tools"
echo "-----------------------"

# PMD (includes CPD - Copy Paste Detector)
if command -v pmd &> /dev/null; then
    installed "pmd (cpd)"
else
    if command -v brew &> /dev/null; then
        installing "pmd via homebrew"
        brew install pmd
    else
        echo "  Install PMD manually: https://pmd.github.io/"
    fi
fi

# ripgrep (for pattern matching)
if command -v rg &> /dev/null; then
    installed "ripgrep"
else
    if command -v brew &> /dev/null; then
        installing "ripgrep via homebrew"
        brew install ripgrep
    else
        echo "  Install ripgrep manually: https://github.com/BurntSushi/ripgrep"
    fi
fi

# =============================================================================
# Summary
# =============================================================================
echo ""
echo "========================================"
echo "🎉 Installation complete!"
echo ""
echo "Available tools:"
echo "  TypeScript/JS: knip, ts-prune, jscpd, eslint"
echo "  Python:        vulture, radon, flake8, pylint"
echo "  Rust:          cargo-udeps, clippy"
echo "  Go:            staticcheck, deadcode"
echo "  Cross-lang:    pmd (cpd), ripgrep"
echo ""
echo "Run '/tech-debt' to scan your codebase!"
