<!-- ABOUTME: Gemini API reference for white paper media generation. -->
<!-- ABOUTME: Documents image models, payloads, and prompt engineering tips. -->
# Gemini Media Generation

## API key handling

- Store the key in `localStorage` under `GEMINI_API_KEY`.
- Never commit keys to the repository.
- The generator panel saves the key locally when you click "Save".

## Image generation (nano banana)

### Text-to-image payload

```json
POST https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-image:generateContent
Headers:
  Content-Type: application/json
  x-goog-api-key: YOUR_API_KEY

{
  "contents": [{
    "role": "user",
    "parts": [{ "text": "Your prompt here" }]
  }],
  "generationConfig": {
    "temperature": 0.2
  }
}
```

### Image-to-image payload

```json
{
  "contents": [{
    "role": "user",
    "parts": [
      {
        "inline_data": {
          "mime_type": "image/png",
          "data": "BASE64_ENCODED_IMAGE"
        }
      },
      { "text": "Your transformation prompt here" }
    ]
  }],
  "generationConfig": {
    "temperature": 0.2
  }
}
```

### Response parsing

```javascript
const partsOut = data?.candidates?.[0]?.content?.parts || [];
const inline = partsOut.find((p) => p.inlineData || p.inline_data);
const inlineData = inline?.inlineData || inline?.inline_data;
const base64 = inlineData?.data;
const mimeType = inlineData?.mimeType || "image/png";
const dataUrl = `data:${mimeType};base64,${base64}`;
```

## Prompt engineering tips

### For diagrams and charts

```
Professional diagram showing [subject], clean minimal lines,
business document aesthetic, white background, [brand palette] accent colors
```

### For figures and illustrations

```
Professional illustration of [subject], clean editorial style,
[brand color] highlights, suitable for corporate white paper
```

### For data visualizations

```
Clean bar chart / line graph / pie chart showing [data concept],
minimal style, [brand colors], white background, no decorative elements
```

## Brand-aware prompts

Each entity profile includes a `mediaPromptPrefix` that is prepended to all prompts:

| Entity | Prefix |
|--------|--------|
| Northwind Labs | "modern tech palette, clean professional style" |
| Apex Consulting | "warm professional palette, refined corporate style" |
| Coastal Biotech | "navy biotech palette, scientific clarity" |

The generator automatically combines the prefix with your prompt:
```
[mediaPromptPrefix], [your prompt]
```

## Using in the template

### Add a generated image

```html
<div class="section-figure">
  <div class="media-placeholder">Loading...</div>
  <img
    class="figure-image gen-media"
    data-media="gemini"
    data-gen="text-to-image"
    data-prompt="Professional diagram of workflow process"
    alt="Figure description"
  />
  <p class="figure-caption">Figure 1: Caption text</p>
</div>
```

### Add an image-to-image transformation

```html
<img
  class="figure-image gen-media"
  data-media="gemini"
  data-gen="image-to-image"
  data-prompt="Restyle with cleaner lines and brand colors"
  alt="Restyled diagram"
/>
```

Note: For image-to-image, load a base image in the generator panel first.

## Generation workflow

1. Press `g` or click "Gen" to open the generator panel
2. Enter your Gemini API key (saved to localStorage)
3. Select the entity profile for brand-aware prompts
4. Click "Generate section" for current section or "Generate all" for entire paper
5. Wait for generation to complete (status shown in panel)

## Best practices for white papers

### Resolution

- Generated images work well on screen
- For print, consider using higher resolution base images with image-to-image
- Test print preview before final export

### Consistency

- Use the same entity profile for all figures
- Maintain consistent prompt structure across images
- Review all generated images together for visual harmony

### Fallbacks

- Keep placeholder text descriptive for accessibility
- Provide meaningful alt text on all images
- Test the document with images disabled

## Supported models

| Model | Use case |
|-------|----------|
| `gemini-2.5-flash-image` | Default for diagrams and illustrations |
| `gemini-2.0-flash-exp` | Alternative image model |

Video generation (Veo) is available but less common for white papers. See the keynote-slides skill for video generation reference.
