#!/usr/bin/env node
/**
 * Narrative Build for White Papers
 *
 * Prepares prompts for model-mediated narrative discovery and paper generation.
 * This script prepares the context; the actual decisions are made by the model.
 *
 * Usage:
 *   node scripts/narrative-build.js papers/<paper-id>
 */

const fs = require('fs');
const path = require('path');

const SCRIPT_DIR = path.dirname(__filename);
const SKILL_DIR = path.dirname(SCRIPT_DIR);
const REFERENCES_DIR = path.join(SKILL_DIR, 'references');
const NARRATIVE_ENGINE_DIR = path.join(REFERENCES_DIR, 'narrative-engine');

function loadReference(filename) {
  const filePath = path.join(NARRATIVE_ENGINE_DIR, filename);
  if (fs.existsSync(filePath)) {
    return fs.readFileSync(filePath, 'utf-8');
  }
  return null;
}

function buildNarrativePrompt(paperPath, ingestion) {
  const paperConfig = ingestion?.paperConfig || {};

  // Load reference materials
  const narrativeArcs = loadReference('narrative-arcs.md');
  const frameworks = loadReference('communication-frameworks.md');
  const frameworkSelection = loadReference('framework-selection.md');
  const checklists = loadReference('checklists.md');

  const prompt = `# White Paper Narrative Discovery

## Context

You are helping build a white paper for: **${paperConfig.title || 'Untitled Paper'}**

Entity: ${paperConfig.entity || 'Not specified'}
Author: ${paperConfig.author || 'Not specified'}

## Materials Available

${ingestion ? formatMaterialsSummary(ingestion) : 'No materials ingested yet. Run ingest-resources.js first.'}

## Discovery Questions

Walk through these questions ONE AT A TIME to understand the paper's needs:

### 1. Audience
Who will read this paper? (e.g., C-Suite executives, technical practitioners, industry analysts, general business readers)

### 2. Purpose
What should readers DO after reading? (e.g., approve a decision, change a process, understand a concept, evaluate options)

### 3. Content Type
What kind of content is this? (e.g., research findings, strategic analysis, technical explanation, market overview, case study)

### 4. Tone
What tone fits the audience and purpose? (e.g., authoritative, provocative, balanced, urgent, visionary)

### 5. Reveal Question
Does the material contain a genuine surprise, counterintuitive finding, or reframe?
- If YES: Consider engagement arcs (Prestige, Mystery Box, Trojan Horse)
- If NO: Consider efficiency frameworks (Pyramid, SCQA, AIDA)

### 6. Density Mode
How much detail does this paper need?
- **Executive**: High-impact, minimal detail, 4-8 pages
- **Standard**: Balanced depth, 8-15 pages
- **Comprehensive**: Full evidence and analysis, 15-25 pages

## Framework Selection

Based on the answers, recommend 2-3 framework combinations from:

${frameworkSelection ? '(See framework-selection.md for full matrix)' : 'Framework selection guide not found'}

## Output Format

After discovery, generate:

1. **draft.md** - Paper outline with:
   - Section titles (H2)
   - Key points for each section
   - Source attribution tags: [PRIMARY], [SECONDARY], [CITED], [SYNTHESIZED], [GENERATED]

2. **narrative-context.json** - Discovery decisions:
   - audience, purpose, contentType, tone, revealPresent
   - selectedFramework (arc + communication framework)
   - densityMode
   - rationale

3. **generation-queue.json** - Visuals needed:
   - type: "svg-chart" | "svg-diagram" | "gemini-image"
   - description, dataSource, placement

## Visual Decision Logic

Use this logic to decide between SVG and Gemini:

| Content | Use SVG | Use Gemini |
|---------|---------|------------|
| Bar/line/pie charts | ✓ | |
| Process workflows | ✓ | |
| Comparison tables | ✓ | |
| Org charts | ✓ | |
| Conceptual illustrations | | ✓ |
| Abstract visualizations | | ✓ |
| Photo-realistic images | | ✓ |
| Diagrams with specific data | ✓ | |

## Reference Materials

The following reference materials are available in ${NARRATIVE_ENGINE_DIR}:

- narrative-arcs.md - 10 engagement-optimized story structures
- communication-frameworks.md - Efficiency frameworks (Pyramid, SCQA, AIDA, etc.)
- framework-selection.md - Selection matrix by content type, audience, purpose
- checklists.md - Quality gates for headlines, CTAs, pricing, factual accuracy

---

Begin the discovery process by asking about the AUDIENCE first.
`;

  return prompt;
}

function formatMaterialsSummary(ingestion) {
  let summary = '';

  if (ingestion.materials?.count > 0) {
    summary += `**${ingestion.materials.count} files** (~${ingestion.materials.totalWordCount} words)\n\n`;

    if (ingestion.summary?.hasBrief) summary += '- Brief document available\n';
    if (ingestion.summary?.hasData) summary += '- Data files available\n';
    if (ingestion.summary?.hasResearch) summary += '- Research documents available\n';
    if (ingestion.summary?.hasNotes) summary += '- Notes/drafts available\n';

    summary += '\nFiles:\n';
    for (const file of ingestion.materials.files.slice(0, 10)) {
      summary += `- ${file.filename} (${file.wordCount || 0} words)\n`;
    }
    if (ingestion.materials.files.length > 10) {
      summary += `- ... and ${ingestion.materials.files.length - 10} more\n`;
    }
  } else {
    summary += 'No materials found in resources/materials/\n';
  }

  if (ingestion.assets?.count > 0) {
    summary += `\n**${ingestion.assets.count} visual assets**\n`;
  }

  return summary;
}

// Main execution
const args = process.argv.slice(2);
const paperPath = args.find((a) => !a.startsWith('--'));

if (!paperPath) {
  console.error('Usage: node scripts/narrative-build.js papers/<paper-id>');
  process.exit(1);
}

if (!fs.existsSync(paperPath)) {
  console.error(`Paper path not found: ${paperPath}`);
  process.exit(1);
}

// Try to load ingestion if it exists
let ingestion = null;
const ingestionPath = path.join(paperPath, 'resources', 'materials', 'ingestion.json');
if (fs.existsSync(ingestionPath)) {
  try {
    ingestion = JSON.parse(fs.readFileSync(ingestionPath, 'utf-8'));
  } catch (e) {
    console.error(`Warning: Could not parse ${ingestionPath}`);
  }
}

// Also try paper.json for config
const configPath = path.join(paperPath, 'paper.json');
if (fs.existsSync(configPath) && !ingestion?.paperConfig) {
  try {
    const config = JSON.parse(fs.readFileSync(configPath, 'utf-8'));
    ingestion = ingestion || {};
    ingestion.paperConfig = config;
  } catch (e) {
    console.error(`Warning: Could not parse ${configPath}`);
  }
}

const prompt = buildNarrativePrompt(paperPath, ingestion);
console.log(prompt);
