#!/bin/bash
# ABOUTME: Bootstrap script to create a new white paper folder with template and config.
# ABOUTME: Usage: new-paper.sh <paper-id> --entity <entity> --title "Paper Title"

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
PROJECT_DIR="$(dirname "$(dirname "$SKILL_DIR")")"
PAPERS_DIR="$PROJECT_DIR/papers"
TEMPLATE="$SKILL_DIR/assets/white-paper.html"

# Parse arguments
PAPER_ID=""
ENTITY="northwind"
TITLE="White Paper"
AUTHOR=""
ORG=""

while [[ $# -gt 0 ]]; do
  case $1 in
    --entity)
      ENTITY="$2"
      shift 2
      ;;
    --title)
      TITLE="$2"
      shift 2
      ;;
    --author)
      AUTHOR="$2"
      shift 2
      ;;
    --org)
      ORG="$2"
      shift 2
      ;;
    -*)
      echo "Unknown option: $1"
      exit 1
      ;;
    *)
      PAPER_ID="$1"
      shift
      ;;
  esac
done

if [ -z "$PAPER_ID" ]; then
  echo "Usage: new-paper.sh <paper-id> [--entity <entity>] [--title \"Title\"] [--author \"Name\"] [--org \"Org\"]"
  exit 1
fi

PAPER_DIR="$PAPERS_DIR/$PAPER_ID"

if [ -d "$PAPER_DIR" ]; then
  echo "Paper directory already exists: $PAPER_DIR"
  exit 1
fi

echo "Creating white paper: $PAPER_ID"
echo "  Entity: $ENTITY"
echo "  Title: $TITLE"

# Create directory structure
mkdir -p "$PAPER_DIR/resources/assets"
mkdir -p "$PAPER_DIR/resources/materials"

# Copy template
cp "$TEMPLATE" "$PAPER_DIR/index.html"

# Create paper-config.js
cat > "$PAPER_DIR/paper-config.js" << EOF
// Paper configuration for $PAPER_ID
window.WHITEPAPER_CONFIG = {
  id: "$PAPER_ID",
  entity: "$ENTITY",
  title: "$TITLE",
  author: "$AUTHOR",
  organization: "$ORG",
  created: "$(date -u +"%Y-%m-%dT%H:%M:%SZ")"
};
EOF

# Create paper.json
cat > "$PAPER_DIR/paper.json" << EOF
{
  "id": "$PAPER_ID",
  "entity": "$ENTITY",
  "title": "$TITLE",
  "author": "$AUTHOR",
  "organization": "$ORG",
  "created": "$(date -u +"%Y-%m-%dT%H:%M:%SZ")",
  "sections": [],
  "status": "draft"
}
EOF

# Create draft.md
cat > "$PAPER_DIR/draft.md" << EOF
# $TITLE

## Working Notes

Add draft content and notes here.

## Outline

1. Executive Summary
2. Introduction
3. Analysis
4. Recommendations
5. Conclusion
6. Appendix

## Key Points

-

## Sources

-
EOF

# Create brief.md
cat > "$PAPER_DIR/resources/materials/brief.md" << EOF
# Paper Brief: $TITLE

## Objective

What is the primary goal of this white paper?

## Target Audience

Who will read this document?

## Key Messages

1.
2.
3.

## Scope

What topics are in/out of scope?

## Sources

What research, data, or materials will inform this paper?

## Timeline

Key milestones and deadlines.
EOF

# Update title in index.html
if [[ "$OSTYPE" == "darwin"* ]]; then
  sed -i '' "s/<title>White Paper<\/title>/<title>$TITLE<\/title>/" "$PAPER_DIR/index.html"
else
  sed -i "s/<title>White Paper<\/title>/<title>$TITLE<\/title>/" "$PAPER_DIR/index.html"
fi

echo ""
echo "Created: $PAPER_DIR"
echo ""
echo "Files:"
echo "  index.html        - Main paper (edit sections here)"
echo "  paper-config.js   - Paper metadata"
echo "  paper.json        - Metadata (JSON)"
echo "  draft.md          - Working notes and outline"
echo "  resources/        - Assets and materials"
echo ""
echo "Next steps:"
echo "  1. Edit draft.md with your outline and notes"
echo "  2. Add materials to resources/materials/"
echo "  3. Edit index.html to build your paper"
echo "  4. Run: scripts/serve-papers.sh to preview"
echo "  5. Open: http://localhost:8922/papers/$PAPER_ID/index.html"
