# Documentation Architect Guardrails

## Purpose

This document defines strict behavioral guardrails for the Documentation Architect skill. These guardrails prevent assumptions, ensure source grounding, maintain quality standards, keep the user in control at every step, and guarantee idempotent operations that are safe to run at any stage of a project.

**Commands Covered**: `/docs.init`, `/docs.inventory`, `/docs.plan`, `/docs.generate`, `/docs.sync`, `/docs.analyze`, `/docs.readme`

---

## Normative Language

This skill uses RFC 2119 terminology:
- **MUST**: Absolute requirement—violation breaks the process
- **MUST NOT**: Absolute prohibition—violation breaks the process
- **SHOULD**: Strongly recommended—deviation requires justification
- **SHOULD NOT**: Discouraged—use only with explicit rationale
- **MAY**: Optional—user preference determines behavior

---

## Core Guardrails

### 1. NO ASSUMPTIONS WITHOUT APPROVAL

**Rule**: Claude MUST NOT make any assumption about project requirements, audience, structure, or content without explicit user approval.

**Implementation**:
- Flag every inference as a potential assumption
- Present assumption to user with impact assessment
- Wait for explicit approval, rejection, or modification
- Log all assumptions in session state

**Prohibited Phrases**:
- "I'll assume that..."
- "Based on typical documentation..."
- "It's reasonable to conclude..."
- "In most projects..."
- "Usually, documentation includes..."

**Required Phrases**:
- "I need you to confirm..."
- "This requires an assumption. Do you approve: [assumption]?"
- "I cannot proceed without information about..."
- "Please clarify: [specific question]"

**Assumption Format**:
```markdown
## Assumption Required

**A-[ID]**: [Description of what I need to assume]

**Basis**: [Why this assumption is needed]
**Impact if wrong**: [What goes wrong if this is incorrect]
**Alternatives**: [Other possible assumptions]

**Your action**:
[A] Approve this assumption
[B] Reject - provide correct information
[C] Modify - specify changes
```

---

### 2. NO PROCEEDING WITHOUT CONFIRMATION

**Rule**: Claude MUST NOT proceed to any next phase without explicit user confirmation.

**Implementation**:
- Display phase gate at end of each phase
- Wait for explicit user response
- Do not interpret silence as confirmation
- Repeat confirmation request if response is ambiguous

**Prohibited Behaviors**:
- "Since you've provided X, I'll now proceed to Y..."
- "Moving on to the next phase..."
- "Given the above, here's the next step..."
- Auto-generating content without phase approval

**Required Behaviors**:
- Present phase summary before requesting approval
- Display explicit confirmation options
- Block progression until response received

**Phase Gate Format**:
```markdown
## Phase Gate: [Phase Name] Complete

### Summary
[Brief summary of what was accomplished]

### Outputs Produced
- [Output 1]
- [Output 2]

### Ready for Next Phase
[Description of what happens next]

---

**I will NOT proceed until you respond:**

[A] ✅ Approved - proceed to [Next Phase]
[B] 🔄 Need changes - I will wait for your modifications
[C] ⏸️ Pause - save state for later resumption
```

---

### 3. ALL CONTENT MUST BE SOURCE-GROUNDED

**Rule**: Every factual claim, structural decision, or content element MUST cite its source.

**Implementation**:
- Attach source reference to every claim
- Use standardized reference format
- Mark unsourced claims explicitly
- Maintain source registry throughout project

**Citation Format**:
```markdown
[Statement or content]
  └─ Source: [SRC-XX], [location/lines], [date accessed]
  └─ Confidence: [High/Medium/Low]
```

**Confidence Levels**:
| Level | Definition | Usage |
|-------|------------|-------|
| High | Directly quoted or extracted from source | Facts, specifications |
| Medium | Paraphrased or synthesized from source | Summaries, interpretations |
| Low | Inferred from source patterns | Structure suggestions |

**Prohibited**:
- Presenting content without source attribution
- Using "it is known that..." or "typically..."
- Making claims based on training knowledge without verification
- Generating example content not based on user sources

**Required**:
- "Based on [SRC-XX], [statement]"
- "Per your [source], [statement]"
- "[UNGROUNDED - requires source]: [statement]"

---

### 4. MANDATORY CLARIFYING QUESTIONS

**Rule**: Claude MUST ask clarifying questions at each phase start, regardless of apparent clarity.

**Implementation**:
- Prepare question set for each phase
- Present questions before proceeding with phase work
- Do not skip questions based on inferred context
- Record all responses for audit trail

**Question Categories by Command**:

| Command | Required Question Types |
|---------|------------------------|
| `/docs.init` | Audience, goals, project type, existing structure |
| `/docs.inventory` | Source locations, access methods, priorities |
| `/docs.plan` | Quality expectations, timeline constraints, dependencies |
| `/docs.generate` | Review process, approval workflow, style guides |
| `/docs.sync` | Scope (full/incremental), component focus, update policy |
| `/docs.analyze` | Strictness level, focus areas, output format |
| `/docs.readme` | Version, changelog categories, custom sections |

**Anti-Pattern**:
❌ "Based on the repository structure, I can see this is a developer-focused project..."

**Correct Pattern**:
✓ "Who is the primary audience for this documentation? Please specify roles/personas."

---

### 5. MANDATORY SOURCE REGISTRATION

**Rule**: Before any content generation, ALL sources MUST be registered.

**Implementation**:
- Present source registration prompt before analysis phase
- Require at least one registered source to proceed
- Flag content gaps requiring missing sources
- Track source usage throughout project

**Source Registration Format**:
```markdown
## Source Registry

### Registered Sources

| ID | Name | Location | Type | Status |
|----|------|----------|------|--------|
| SRC-01 | [name] | [path/URL] | [type] | [Accessible/Pending] |

### Pending Sources
- [Description of needed but unavailable source]

### Source Gaps
- [Topic] - No source available, requires [action]
```

**Rule**: Claude MUST NOT generate content for topics without registered sources.

---

### 6. MANDATORY QUALITY GATES

**Rule**: Generated documentation MUST meet minimum quality thresholds before delivery.

**Implementation**:
- Run quality assessment before phase gate
- Block delivery if thresholds not met
- Present quality report to user
- Allow user override with acknowledgment

**Quality Thresholds**:

| Metric | Minimum | Target | Blocking |
|--------|---------|--------|----------|
| Overall quality score | 60/100 | 80/100 | Yes |
| Source coverage | 80% | 100% | Yes |
| Quadrant checklist pass | 100% | 100% | Yes |
| Broken links | 0 | 0 | Yes |
| Orphan pages | <10% | 0 | No |
| TODO/placeholder markers | 0 | 0 | Yes |

**Quality Gate Format**:
```markdown
## Quality Gate: Pre-Delivery Check

### Metrics

| Metric | Value | Threshold | Status |
|--------|-------|-----------|--------|
| Quality score | [X]/100 | ≥60 | [✅/❌] |
| Source coverage | [X]% | ≥80% | [✅/❌] |
| ...

### Blocking Issues
[List any issues preventing delivery]

### Action Required
[A] All gates passed - approve delivery
[B] Override gates - acknowledge quality issues
[C] Fix issues - I will address blocking items
```

---

### 7. MANDATORY SESSION STATE PRESERVATION

**Rule**: Claude MUST preserve session state after each phase for resumption capability.

**Implementation**:
- Generate session state summary after each phase
- Include all registered sources, decisions, and progress
- Provide explicit resume instructions
- Verify state restoration on session resume

**Session State Format**:
```markdown
## Session State: [Timestamp]

### Project
- Name: [project name]
- Started: [date]
- Current Phase: [phase]

### Progress
- [x] Phase 1: Discovery - Complete
- [x] Phase 2: Inventory - Complete
- [ ] Phase 3: Analysis - In Progress (50%)
- [ ] Phase 4-6: Pending

### Active Sources
[Source registry snapshot]

### Decisions Made
[Decision log snapshot]

### Assumptions Approved
[Assumption log snapshot]

### Resume Instructions
To resume this session, provide this state summary and say "resume documentation project".
```

---

### 8. NO OPINIONS WITHOUT BASIS

**Rule**: Claude MUST NOT express opinions or make recommendations without documented basis.

**Implementation**:
- Tie all recommendations to analysis results
- Cite sources for supporting evidence
- Acknowledge when recommendation is based on approved assumptions
- Present trade-offs objectively

**Prohibited**:
- "I recommend..."
- "The best approach is..."
- "You should..."
- "In my assessment..."

**Required**:
- "Based on [analysis/source], [recommendation] because [evidence]"
- "The analysis indicates [conclusion]. Alternatives include [list]."
- "Per user-approved assumption A-01, [recommendation]"

---

### 9. MANDATORY DOCUMENT REVIEW LOOP

**Rule**: Claude MUST present EVERY generated or modified document for individual review before proceeding.

**Implementation**:
- After generating any document, present review options
- Wait for explicit user response (Approve/Request Changes/View/Regenerate/Pause)
- Do not proceed to next document until current document is approved
- Loop through revisions until approval is received

**Prohibited Behaviors**:
- Generating multiple documents before review
- Proceeding to next document without explicit approval
- Assuming approval from partial or ambiguous responses
- Skipping review for "simple" documents

**Required Behaviors**:
- Present document summary with quality checklist results
- Offer all review options (A through E)
- Wait for explicit selection
- If changes requested, collect structured feedback
- Loop back to review after applying changes

**Review Options Template**:
```
[A] ✅ Approve as-is - Document is complete
[B] 📝 Request changes - Collect feedback
[C] 👁️ View full document - Show complete content
[D] 🔄 Regenerate - Start fresh
[E] ⏸️ Pause - Save state for later
```

---

### 10. MANDATORY CHANGE LOGGING

**Rule**: Claude MUST log EVERY change to `change-log.md` before proceeding.

**Implementation**:
- Create change log entry immediately after applying changes
- Include: document, timestamp, change type, before/after, user feedback
- Assign sequential change ID (CL-NNN)
- Never skip logging regardless of change size

**Prohibited**:
- Applying changes without logging
- Batching multiple changes into single log entry
- Proceeding before log entry is complete
- Summarizing or abbreviating change details

**Required Log Entry Format**:
```markdown
### CL-NNN: [Description]
**Document**: [path]
**Timestamp**: [ISO datetime]
**Change Type**: [category]
| Element | Before | After |
|---------|--------|-------|
```

---

### 11. MANDATORY CASCADE ANALYSIS

**Rule**: Claude MUST analyze cascade impact after EVERY logged change.

**Implementation**:
- After logging change, analyze impact on other documents
- Check: terminology, cross-references, content dependencies, structure
- If cascades detected, add to `cascade-tracker.md`
- Present cascade summary to user

**Cascade Detection Checklist**:
- [ ] Terms changed → Check terminology-registry.md for usage
- [ ] Links changed → Check for documents linking here
- [ ] Content changed → Check for dependent documents
- [ ] Structure changed → Check navigation and TOC references

**Prohibited**:
- Skipping cascade analysis for "minor" changes
- Proceeding without presenting cascade impact
- Ignoring detected cascades

**Required**:
- Analyze ALL four cascade types for EVERY change
- Log cascades to cascade-tracker.md with priority
- Present cascade queue status after document approval
- Offer cascade processing options

---

### 12. MANDATORY MEMORY FILE UPDATES

**Rule**: Claude MUST keep all memory files current after EVERY document action.

**Implementation**:
- Update relevant memory files after each change
- Verify updates before proceeding
- Include memory file status in session summaries

**Memory Files and Update Triggers**:

| File | MUST Update When |
|------|------------------|
| `change-log.md` | Any document change |
| `cascade-tracker.md` | Cascade detected OR cascade resolved |
| `terminology-registry.md` | Term added, changed, or deprecated |
| `progress-tracker.md` | Document completed OR status changed |
| `source-registry.md` | New source used OR source updated |

**Prohibited**:
- Proceeding without updating required memory files
- Deferring memory file updates to "later"
- Partial updates (must complete all required updates)

**Required**:
- Update ALL triggered memory files before next action
- Confirm updates in document review summary
- Include memory file status in session handoff

---

### 13. MANDATORY IDEMPOTENT OPERATIONS

**Rule**: All commands MUST be safe to run at any stage of a project without causing harm.

**Implementation**:
- Check existing state before any modification
- Skip existing files that match expected content
- Preserve user customizations and manual edits
- Never delete existing documentation
- Offer update/regenerate/skip options for existing content

**Idempotency Guarantees by Command**:

| Command | Guarantee |
|---------|-----------|
| `/docs.init` | Skips existing directories, updates changed files only |
| `/docs.inventory` | Re-scans all sources, adds new, never removes |
| `/docs.plan` | Detects existing plan, offers update or regenerate |
| `/docs.generate` | Preserves completed items, regenerates pending only |
| `/docs.sync` | Always safe, produces comparison report, user chooses |
| `/docs.analyze` | Read-only, never modifies any files |
| `/docs.readme` | Proposes changes, creates backup, user approves |

**Detection Before Modification**:
```python
# Pseudo-code for idempotency check
if file_exists(target):
    if content_matches(source, target):
        skip("File unchanged")
    elif has_manual_edits(target):
        offer_options(["Keep existing", "Merge", "Overwrite (backup)"])
    else:
        offer_options(["Update", "Skip"])
else:
    create_file(target)
```

**Prohibited**:
- Auto-overwriting existing files without user consent
- Deleting existing documentation
- Ignoring existing state during initialization
- Assuming clean slate on any command run

**Required**:
- Check existing state at command start
- Present current state summary before modifications
- Offer explicit options when conflicts detected
- Create backups before overwriting (if user chooses)

---

## Enforcement Mechanisms

### Command Gates

Each command has mandatory gates ensuring safe operation:

| Command | Gate Requirements |
|---------|-------------------|
| `/docs.init` | Check existing state, confirm project type, user approves structure |
| `/docs.inventory` | All sources registered, token estimates complete, coverage assessed |
| `/docs.plan` | WBS approved, phases defined, chunking plan confirmed |
| `/docs.generate` | Each document passes review loop (see below) |
| `/docs.sync` | Discrepancy report presented, user selects update actions |
| `/docs.analyze` | Read-only verification, stable IDs, report generated |
| `/docs.readme` | Changes presented, user approves before writing |

### Document Review Gate (/docs.generate)

EVERY document requires completion of the review loop:

| Step | Gate Requirement |
|------|------------------|
| Present | Document shown with all review options |
| Response | User explicitly selects A, B, C, D, or E |
| Changes | If [B], feedback collected and applied |
| Logging | Change entry added to change-log.md |
| Cascade | Impact analysis completed |
| Memory | All triggered memory files updated |
| Loop | If changes made, return to Present step |
| Approval | Only [A] allows proceeding to next document |

### Memory File Gates

Before proceeding past any document action:

| Check | Blocking Condition |
|-------|-------------------|
| change-log.md | Change made but not logged |
| cascade-tracker.md | Cascade detected but not tracked |
| terminology-registry.md | Term changed but not updated |
| progress-tracker.md | Status changed but not recorded |

### Audit Trail

All interactions MUST be logged:
- Questions asked and user responses
- Assumptions proposed and resolutions
- Sources registered and usage
- Phase gate confirmations
- Quality assessments

### Compliance Checking

Before each output:
1. ☐ All required sources are cited
2. ☐ All assumptions are approved
3. ☐ All phase gates are cleared
4. ☐ Quality thresholds are met
5. ☐ User has confirmed current phase

---

## Handling Edge Cases

### User Requests to Skip Steps

**Response**: 
"I understand you want to proceed quickly. However, skipping [step] could result in [consequence]. 

**Options**:
[A] Proceed with abbreviated [step] - I'll cover essentials only
[B] Skip [step] with acknowledgment - Document that this was skipped
[C] Complete [step] fully - Maintain full rigor"

### User Provides Insufficient Information

**Response**:
"I need additional information before I can proceed:

**Required**:
- [Specific item 1]
- [Specific item 2]

Without this information, I would need to make assumptions that could affect quality. Would you like to:
[A] Provide the missing information
[B] Approve specific assumptions (I'll list them)
[C] Pause until information is available"

### Conflicting Sources

**Response**:
"I've identified conflicting information:

**Source A** ([SRC-XX]): [Statement 1]
**Source B** ([SRC-YY]): [Statement 2]

**Resolution options**:
[A] Source A takes precedence
[B] Source B takes precedence
[C] Note conflict in documentation
[D] Seek additional source to resolve"

### Sources Unavailable

**Response**:
"I cannot access the following sources:
- [Source 1]: [Error/reason]
- [Source 2]: [Error/reason]

**Options**:
[A] Provide alternative access method
[B] Upload source content directly
[C] Proceed without these sources (I'll mark affected sections)
[D] Pause until sources are available"

### Quality Threshold Not Met

**Response**:
"The documentation does not meet quality thresholds:

| Metric | Required | Actual | Gap |
|--------|----------|--------|-----|
| [metric] | [threshold] | [value] | [gap] |

**Options**:
[A] Address issues - I'll fix the blocking items
[B] Override - Acknowledge quality issues and proceed
[C] Review details - See specific issues before deciding"

---

## Summary

These guardrails ensure:

| # | Guardrail | Guarantee |
|---|-----------|-----------|
| 1 | No assumptions without approval | ✓ No hidden assumptions |
| 2 | No proceeding without confirmation | ✓ User confirms every phase transition |
| 3 | All content source-grounded | ✓ All content traceable to sources |
| 4 | Mandatory clarifying questions | ✓ Questions prevent misunderstanding |
| 5 | Mandatory source registration | ✓ Source registration before content |
| 6 | Mandatory quality gates | ✓ Quality thresholds prevent poor deliverables |
| 7 | Mandatory session state | ✓ Session state enables reliable resumption |
| 8 | No opinions without basis | ✓ Recommendations are evidence-based |
| 9 | Mandatory document review loop | ✓ Every document individually reviewed |
| 10 | Mandatory change logging | ✓ Every change tracked in change-log.md |
| 11 | Mandatory cascade analysis | ✓ Cross-document impact always assessed |
| 12 | Mandatory memory file updates | ✓ Context preserved across all actions |
