---
name: documentation-architect
description: >
  Transform documentation from any starting point into professional, comprehensive documentation
  packages using the Diátaxis framework. 7 commands: init (create structure), inventory (catalog sources),
  plan (create WBS), generate (create docs), sync (update from code reality), analyze (quality audit),
  readme (manage README/CHANGELOG). Integrates with speckit-generator for implementation-to-docs workflow.
  Supports specs, ADRs, RFCs as input with code walkthrough for syncing docs to reality.
---

# Documentation Architect

Create comprehensive, professional documentation using the Diátaxis framework. Works with any starting point: greenfield projects, existing specs/ADRs/RFCs, or scattered documentation.

## Critical: Read GUARDRAILS.md First

Before any command, Claude **MUST** read and internalize the behavioral constraints in `GUARDRAILS.md`.

**Key Guardrails Summary**:
| # | Guardrail | Brief |
|---|-----------|-------|
| 1 | No assumptions without approval | Every inference requires confirmation |
| 2 | No proceeding without confirmation | User controls progression |
| 3 | Source-grounded content | Every claim cites its source |
| 4 | Mandatory document review loop | Every document individually reviewed |
| 5 | Mandatory change logging | All changes tracked |
| 6 | Mandatory cascade analysis | Cross-document impact assessed |
| 7 | Idempotent operations | Safe to run at any project stage |

---

## Commands

| Command | Purpose | When to Use |
|---------|---------|-------------|
| `/docs.init` | Create docs/ structure and memory files | Starting documentation |
| `/docs.inventory` | Catalog and classify sources | After init or adding sources |
| `/docs.plan` | Create documentation WBS | Before generating docs |
| `/docs.generate` | Execute plan, create docs | When plan is ready |
| `/docs.sync` | Walk code, sync with reality | After implementation |
| `/docs.analyze` | Quality audit (read-only) | Before release, CI/CD |
| `/docs.readme` | Manage README.md, CHANGELOG.md | Project root files |

---

## Quick Start

```
New project:
  /docs.init → /docs.inventory → /docs.plan → /docs.generate

After speckit implementation:
  /docs.sync → review discrepancies → update docs

Quality check before release:
  /docs.analyze → address findings → /docs.readme --changelog VERSION
```

---

## The Diátaxis Framework

Documentation organized around four user needs:

```
                PRACTICAL                      THEORETICAL
          ┌─────────────────────────────┬─────────────────────────────┐
          │                             │                             │
LEARNING  │      TUTORIALS              │      EXPLANATION            │
          │  "Help me learn"            │  "Help me understand why"   │
          │                             │                             │
          ├─────────────────────────────┼─────────────────────────────┤
          │                             │                             │
WORKING   │      HOW-TO GUIDES          │      REFERENCE              │
          │  "Help me do X"             │  "Give me the facts"        │
          │                             │                             │
          └─────────────────────────────┴─────────────────────────────┘
```

See `references/diataxis-framework.md` for detailed guidance.

---

## /docs.init

**Purpose**: Establish documentation foundation.

**Workflow**:
1. Check existing state (docs/, .claude/memory/docs-*.md)
2. Detect project type (library, CLI, app, service)
3. Create docs/ structure with Diátaxis layout
4. Create documentation memory files
5. Present summary and next steps

**Outputs**:
```
docs/
├── index.md
├── user/
│   ├── getting-started/
│   ├── guides/
│   ├── concepts/
│   └── reference/
├── developer/
│   ├── architecture/
│   ├── contributing/
│   └── reference/api/
└── _meta/
    ├── inventory.md
    ├── plan.md
    └── progress.md

.claude/memory/
├── docs-constitution.md
├── docs-terminology.md
└── docs-sources.md
```

**Integration**: Auto-suggested after `/speckit.init`

See `references/command-workflows/init-workflow.md` for details.

---

## /docs.inventory

**Purpose**: Catalog and classify documentation sources.

**Workflow**:
1. Scan source locations (.claude/resources/, codebase, uploads)
2. Classify by type (SPEC, ADR, RFC, CODE, DOC)
3. Map to Diátaxis quadrants
4. Estimate token counts for chunking
5. Identify coverage gaps

**Source Types**:
| Type | Description | Example |
|------|-------------|---------|
| SPEC | Requirements, features | requirements.md, PRD |
| ADR | Architecture decisions | ADR-001-auth.md |
| RFC | Proposals, designs | RFC-002-api.md |
| CODE | Docstrings, comments | Python/TS files |
| DOC | Existing documentation | README, guides |

**Outputs**: `docs/_meta/inventory.md`, updated `docs-sources.md`

See `references/command-workflows/inventory-workflow.md` for details.

---

## /docs.plan

**Purpose**: Create documentation plan (Work Breakdown Structure).

**Modes**:
- `/docs.plan` - Plan from inventory
- `/docs.plan --from-speckit` - Use speckit artifacts as input

**Workflow**:
1. Load inventory
2. Analyze sources for documentation needs
3. Map to Diátaxis quadrants
4. Design target structure
5. Create prioritized WBS
6. Define phases and gates

**WBS Item Structure**:
```markdown
| ID | Document | Quadrant | Priority | Sources | Dependencies |
|----|----------|----------|----------|---------|--------------|
| WBS-001 | quickstart.md | Tutorial | HIGH | SRC-001 | None |
| WBS-002 | authentication.md | How-To | HIGH | SRC-002,SRC-003 | WBS-001 |
```

**Outputs**: `docs/_meta/plan.md`

See `references/command-workflows/plan-workflow.md` for details.

---

## /docs.generate

**Purpose**: Execute plan to create documentation.

**Selection Options**:
- `/docs.generate` - All pending items
- `/docs.generate WBS-001` - Specific item
- `/docs.generate "Phase 1"` - Items in phase
- `/docs.generate --user` - User docs only
- `/docs.generate --dev` - Developer docs only

**Workflow**:
1. Load plan WBS items
2. Resolve dependencies
3. For each item:
   - Load sources
   - Apply Diátaxis guidelines
   - Generate document
   - **Document Review Loop** (mandatory)
   - Log changes
   - Cascade analysis
   - Update memory files
4. Phase gate check

**Document Review Loop** (per document):
```
Generate → Present Review → Collect Feedback
                               ↓
                    [Approved] → Update Memory → Next
                    [Changes] → Apply → Log → Cascade → Re-present
```

**Outputs**: Generated docs in docs/user/ and docs/developer/

See `references/command-workflows/generate-workflow.md` for details.

---

## /docs.sync

**Purpose**: Walk codebase, sync documentation with implementation reality.

**Key Insight**: Documentation drifts from reality during implementation. This command bridges that gap by treating code as source of truth.

**Modes**:
- `/docs.sync` - Incremental sync (changed files)
- `/docs.sync --walkthrough` - Full code exploration
- `/docs.sync --component auth` - Specific component

**Workflow**:
1. Code walkthrough (analyze implementation)
2. Extract reality (APIs, configs, behavior)
3. Compare to existing documentation
4. Generate discrepancy report
5. Present update options per finding:
   - **Auto-update**: Apply simple fixes
   - **Manual review**: Complex changes
   - **Skip**: Acknowledge, don't change
   - **Code issue**: Doc is correct, code needs fix
6. Update memory files with code snapshot

**Discrepancy Types**:
| Type | Severity | Example |
|------|----------|---------|
| MISSING | HIGH | Public API not documented |
| INCORRECT | HIGH | Doc says 201, code returns 200 |
| OUTDATED | MEDIUM | References deprecated endpoint |
| UNDOCUMENTED | MEDIUM | Public function lacks docstring |

**Outputs**:
- `docs/_meta/sync-report.md`
- `.claude/memory/docs-codebase-snapshot.md`
- Updated documentation

**Integration**: Suggested after `/speckit.implement`

See `references/command-workflows/sync-workflow.md` for details.

---

## /docs.analyze

**Purpose**: Read-only audit of documentation quality.

**Characteristics**:
- **Read-only**: Never modifies files
- **Deterministic**: Same input = same output
- **Stable IDs**: Finding IDs consistent across runs

**Checks**:
- Document quality scores
- Diátaxis coverage matrix
- Broken links, orphan pages
- TODO/placeholder markers
- User journey coverage

**Quality Metrics**:
| Metric | Weight | Criteria |
|--------|--------|----------|
| Accuracy | 25% | Claims verified, sources cited |
| Clarity | 25% | Scannable, examples present |
| Completeness | 25% | All sections, no TODOs |
| Structure | 25% | Follows quadrant template |

**Outputs**: `docs/_meta/analysis-report.md`

See `references/command-workflows/analyze-workflow.md` for details.

---

## /docs.readme

**Purpose**: Direct management of README.md and CHANGELOG.md.

**Modes**:
- `/docs.readme` - Update README.md
- `/docs.readme --init` - Create from scratch
- `/docs.readme --changelog VERSION` - Add changelog entry

**README Best Practices**:
```markdown
# Project Name
> One-line description

## Installation
## Quick Start
## Features
## Documentation (links to docs/)
## Contributing
## License
```

**CHANGELOG Format** (Keep a Changelog):
```markdown
## [Unreleased]
### Added / Changed / Fixed

## [1.0.0] - YYYY-MM-DD
### Added
- Initial release
```

**Outputs**: README.md, CHANGELOG.md (with user review)

See `references/command-workflows/readme-workflow.md` for details.

---

## speckit-generator Integration

| After speckit Command | Suggested docs Action |
|----------------------|----------------------|
| `/speckit.init` | `/docs.init` |
| `/speckit.plan` | `/docs.plan --from-speckit` |
| `/speckit.tasks` | `/docs.inventory` |
| `/speckit.implement` | `/docs.sync` |

See `references/speckit-integration.md` for integration details.

---

## docs/ Output Structure

```
docs/
├── index.md                    # Landing page
│
├── user/                       # USER-FACING
│   ├── getting-started/        # Tutorials
│   │   ├── quickstart.md
│   │   └── installation.md
│   ├── guides/                 # How-To
│   │   └── [task].md
│   ├── concepts/               # Explanations
│   │   └── [concept].md
│   └── reference/              # Reference
│       └── configuration.md
│
├── developer/                  # DEVELOPER-FACING
│   ├── architecture/           # Design, ADRs
│   │   ├── overview.md
│   │   └── decisions/
│   ├── contributing/           # Contribution guides
│   │   └── getting-started.md
│   └── reference/              # API/Technical
│       └── api/
│
└── _meta/                      # Metadata (not published)
    ├── inventory.md
    ├── plan.md
    ├── progress.md
    ├── change-log.md
    ├── sync-report.md
    └── analysis-report.md
```

---

## Memory Files

| File | Purpose | Created By |
|------|---------|------------|
| `docs-constitution.md` | Documentation principles | `/docs.init` |
| `docs-sources.md` | Source registry | `/docs.inventory` |
| `docs-terminology.md` | Term definitions | `/docs.generate` |
| `docs-codebase-snapshot.md` | Code state snapshot | `/docs.sync` |

---

## Idempotency Guarantees

| Command | Behavior |
|---------|----------|
| `/docs.init` | Skips existing, updates changed, preserves customizations |
| `/docs.inventory` | Re-scans, updates registry, adds new, never removes |
| `/docs.plan` | Detects existing, offers update/regenerate |
| `/docs.generate` | Preserves completed, regenerates pending |
| `/docs.sync` | Always safe, produces comparison, user chooses |
| `/docs.analyze` | Read-only, stable IDs |
| `/docs.readme` | Proposes changes, user approves |

---

## Scripts

| Script | Purpose | Used By |
|--------|---------|---------|
| `scripts/analyze_docs.py` | Quality analysis | `/docs.analyze` |
| `scripts/doc_research.py` | Research competitors | `/docs.inventory` |
| `scripts/generate_wbs.py` | Generate WBS | `/docs.plan` |
| `scripts/validate_docs.py` | Validation checks | `/docs.analyze` |
| `scripts/sync_codebase.py` | Code walkthrough | `/docs.sync` |
| `scripts/readme_generator.py` | README/CHANGELOG | `/docs.readme` |

---

## References

| Document | Purpose |
|----------|---------|
| `GUARDRAILS.md` | Behavioral constraints |
| `references/diataxis-framework.md` | Quadrant guidelines |
| `references/chunking-strategy.md` | Large source handling |
| `references/source-tracking.md` | Citation protocols |
| `references/quality-criteria.md` | Assessment rubrics |
| `references/speckit-integration.md` | speckit workflow |
| `references/command-workflows/*.md` | Command details |

---

## Anti-Patterns

| Anti-Pattern | Correct Approach |
|--------------|------------------|
| Mixed quadrant content | Separate by user need |
| No quickstart | 5-minute first experience |
| Scattered explanations | Dedicated concepts section |
| Reference-only docs | Full quadrant coverage |
| Docs drift from code | Use `/docs.sync` after implementation |
| Generating without sources | Register sources first |
| Proceeding without confirmation | Wait for user approval |
