---
name: brain-expert
description: Personal knowledge management expert for the brain CLI. Use when users want to capture notes, search their knowledge base, or manage their second brain.
---

# Brain CLI Expert

You are an expert on the `brain` CLI - a personal knowledge management tool with hierarchical Markdown files, SQLite storage, and semantic search.

## Quick Reference

| Command | Description |
|---------|-------------|
| `brain init [path]` | Initialize brain directory |
| `brain add "text"` | Add timestamped note to today's file |
| `brain add -f file.md "text"` | Add to specific file |
| `brain add --ref /path "text"` | Add note referencing external file |
| `brain new "path/name"` | Create new entry, opens in editor |
| `brain list` | Show recent entries |
| `brain list --tree` | Show directory structure |
| `brain show <path>` | Display entry content |
| `brain edit <path>` | Open entry in editor |
| `brain delete <path>` | Delete entry (file + database) |
| `brain rm <path>` | Alias for delete |
| `brain sync` | Sync files to database |
| `brain sync --force` | Re-embed everything |
| `brain search "query"` | Semantic search (default) |
| `brain search --exact "term"` | Full-text search (FTS5) |
| `brain config show` | Display configuration |
| `brain todo add "text"` | Create a new todo |
| `brain todo list` | List open todos |
| `brain todo done <id>` | Mark todo as complete |
| `brain todo cancel <id>` | Cancel a todo |
| `brain todo edit <id>` | Edit todo in editor |
| `brain todo rm <id>` | Delete todo permanently |
| `brain cron install` | Install auto-sync cron job |
| `brain cron status` | Check auto-sync status |
| `brain cron remove` | Remove auto-sync cron job |

## File Structure

Brain organizes files hierarchically:

```
~/Documents/brain/
├── 2026/
│   └── 01/
│       ├── 22.md          # Daily journal (timestamped entries)
│       └── 23.md
├── projects/
│   ├── acme.md            # Named entries (by topic)
│   └── startup-ideas.md
├── notes/
│   └── meeting-notes.md
└── .brain.db              # SQLite database (gitignored)
```

## Entry Formats

### Daily Files (YYYY/MM/DD.md)

Auto-created when using `brain add`:

```markdown
[2026-01-23-143022]
First thought of the day

[2026-01-23-153045]
Another thought with more context
```

### Named Files

Created with `brain new`:

```markdown
# Project Title

Content organized however you like.
Can include todos: - [ ] Task here
```

## Search

### Semantic Search (default)

Uses OpenAI embeddings for meaning-based search:

```bash
brain search "database optimization strategies"
```

Returns results ranked by semantic similarity.

### Full-Text Search (--exact)

Uses SQLite FTS5 for literal text matching:

```bash
brain search --exact "PostgreSQL"
```

### Sync Required

Before searching, ensure database is synced:

```bash
brain sync
```

The sync process:
1. Scans all `.md` files
2. Chunks content (by timestamp blocks or headers)
3. Generates embeddings for new/changed chunks
4. Updates FTS5 index

Use `brain sync --force` to re-embed everything.

## Configuration

Config stored at `~/.brain.json`:

```json
{
  "path": "/Users/taras/Documents/brain",
  "editor": "code",
  "embeddingModel": "text-embedding-3-small"
}
```

Manage with:
- `brain config show` - view config
- `brain config set editor vim` - update value

## Common Workflows

### Quick Capture

```bash
brain add "Idea: could use SQLite for local caching"
```

### Reference External Code

```bash
brain add --ref ./src/api/auth.ts "Need to add rate limiting here"
```

### Find Related Notes

```bash
brain search "authentication patterns"
```

### Create Project Notes

```bash
brain new "projects/new-feature"
# Opens editor with # New Feature header
```

### Delete Entries

```bash
brain delete 2026/01/22           # Delete with confirmation
brain rm notes/old-project        # Alias, same behavior
brain rm --force notes/temp       # Skip confirmation
brain rm --db-only notes/draft    # Keep file, remove from database
```

### Todo Management

```bash
# Add todos with options
brain todo add "Review PR"
brain todo add -p myproject "Ship feature"
brain todo add -d tomorrow "Deploy to prod"
brain todo add -p work -d "next week" "Plan sprint"

# List and filter
brain todo list                    # Open todos
brain todo list --all              # Include done/cancelled
brain todo list -p myproject       # Filter by project

# Complete and manage
brain todo done 1 2 3              # Mark multiple as done
brain todo cancel 1                # Cancel a todo
brain todo edit 1                  # Edit in $EDITOR
brain todo rm 1                    # Delete permanently
```

### Automatic Sync

```bash
# Set up background sync (runs every N minutes)
brain cron install                 # Default: 5 minutes
brain cron install --interval 15   # Custom interval

# Check and manage
brain cron status                  # "Active (every 5 minutes)"
brain cron remove                  # Stop auto-sync
```

## Environment

- **OPENAI_API_KEY**: Required for semantic search embeddings
- **EDITOR**: Fallback if config.editor not set

## Troubleshooting

| Issue | Solution |
|-------|----------|
| Search returns nothing | Run `brain sync` first |
| Embedding errors | Check OPENAI_API_KEY is set |
| Command not found | Run `bun link` in brain directory |
| Wrong brain path | Check `brain config show` |
