#!/usr/bin/env python3
"""
Content Similarity Checker - Compare documents for similarity.
"""

import argparse
import json
import os
import re
from typing import Dict, List, Optional, Tuple
from pathlib import Path

import numpy as np
import pandas as pd
from sklearn.feature_extraction.text import TfidfVectorizer, CountVectorizer
from sklearn.metrics.pairwise import cosine_similarity

try:
    import nltk
    from nltk.tokenize import word_tokenize
    from nltk.corpus import stopwords
    NLTK_AVAILABLE = True
except ImportError:
    NLTK_AVAILABLE = False


class SimilarityChecker:
    """Compare documents and text for similarity."""

    def __init__(self, method: str = "cosine"):
        """
        Initialize the similarity checker.

        Args:
            method: Comparison method ("cosine", "jaccard", "levenshtein", "tfidf")
        """
        self.method = method
        self._vectorizer = None
        self._stop_words = None

        if NLTK_AVAILABLE:
            try:
                self._stop_words = set(stopwords.words('english'))
            except LookupError:
                try:
                    nltk.download('stopwords', quiet=True)
                    nltk.download('punkt', quiet=True)
                    self._stop_words = set(stopwords.words('english'))
                except Exception:
                    pass

    def _preprocess(self, text: str) -> str:
        """Preprocess text for comparison."""
        # Lowercase
        text = text.lower()
        # Remove special characters
        text = re.sub(r'[^\w\s]', ' ', text)
        # Remove extra whitespace
        text = ' '.join(text.split())
        return text

    def _tokenize(self, text: str) -> List[str]:
        """Tokenize text into words."""
        text = self._preprocess(text)

        if NLTK_AVAILABLE:
            try:
                tokens = word_tokenize(text)
            except Exception:
                tokens = text.split()
        else:
            tokens = text.split()

        # Remove stopwords if available
        if self._stop_words:
            tokens = [t for t in tokens if t not in self._stop_words]

        return tokens

    def compare(self, text1: str, text2: str) -> float:
        """
        Compare two texts for similarity.

        Args:
            text1: First text
            text2: Second text

        Returns:
            Similarity score (0.0 to 1.0)
        """
        if self.method == "cosine":
            return self._cosine_similarity(text1, text2)
        elif self.method == "jaccard":
            return self._jaccard_similarity(text1, text2)
        elif self.method == "levenshtein":
            return self._levenshtein_similarity(text1, text2)
        elif self.method == "tfidf":
            return self._tfidf_similarity(text1, text2)
        else:
            raise ValueError(f"Unknown method: {self.method}")

    def _cosine_similarity(self, text1: str, text2: str) -> float:
        """Calculate cosine similarity using term frequencies."""
        vectorizer = CountVectorizer()
        try:
            vectors = vectorizer.fit_transform([text1, text2])
            similarity = cosine_similarity(vectors[0:1], vectors[1:2])[0][0]
            return float(similarity)
        except ValueError:
            return 0.0

    def _tfidf_similarity(self, text1: str, text2: str) -> float:
        """Calculate similarity using TF-IDF."""
        vectorizer = TfidfVectorizer()
        try:
            vectors = vectorizer.fit_transform([text1, text2])
            similarity = cosine_similarity(vectors[0:1], vectors[1:2])[0][0]
            return float(similarity)
        except ValueError:
            return 0.0

    def _jaccard_similarity(self, text1: str, text2: str) -> float:
        """Calculate Jaccard similarity (set intersection over union)."""
        tokens1 = set(self._tokenize(text1))
        tokens2 = set(self._tokenize(text2))

        if not tokens1 and not tokens2:
            return 1.0
        if not tokens1 or not tokens2:
            return 0.0

        intersection = len(tokens1 & tokens2)
        union = len(tokens1 | tokens2)

        return intersection / union if union > 0 else 0.0

    def _levenshtein_similarity(self, text1: str, text2: str) -> float:
        """Calculate normalized Levenshtein similarity."""
        s1 = self._preprocess(text1)
        s2 = self._preprocess(text2)

        if not s1 and not s2:
            return 1.0
        if not s1 or not s2:
            return 0.0

        # Calculate Levenshtein distance
        len1, len2 = len(s1), len(s2)

        if len1 > len2:
            s1, s2 = s2, s1
            len1, len2 = len2, len1

        current_row = range(len1 + 1)

        for i in range(1, len2 + 1):
            previous_row, current_row = current_row, [i] + [0] * len1
            for j in range(1, len1 + 1):
                add, delete, change = previous_row[j] + 1, current_row[j-1] + 1, previous_row[j-1]
                if s1[j-1] != s2[i-1]:
                    change += 1
                current_row[j] = min(add, delete, change)

        distance = current_row[len1]
        max_len = max(len1, len2)

        return 1 - (distance / max_len) if max_len > 0 else 1.0

    def compare_files(self, file1: str, file2: str) -> float:
        """
        Compare two files for similarity.

        Args:
            file1: Path to first file
            file2: Path to second file

        Returns:
            Similarity score
        """
        with open(file1, 'r', encoding='utf-8', errors='ignore') as f:
            text1 = f.read()
        with open(file2, 'r', encoding='utf-8', errors='ignore') as f:
            text2 = f.read()

        return self.compare(text1, text2)

    def compare_all_methods(self, text1: str, text2: str) -> Dict:
        """
        Compare using all available methods.

        Args:
            text1: First text
            text2: Second text

        Returns:
            Dictionary with scores from each method
        """
        results = {
            "cosine": self._cosine_similarity(text1, text2),
            "jaccard": self._jaccard_similarity(text1, text2),
            "levenshtein": self._levenshtein_similarity(text1, text2),
            "tfidf": self._tfidf_similarity(text1, text2)
        }

        results["average"] = sum(results.values()) / len(results)
        return results

    def compare_to_corpus(self, text: str, corpus: List[str]) -> List[Dict]:
        """
        Compare text against a corpus of documents.

        Args:
            text: Query text
            corpus: List of documents

        Returns:
            List of similarity results, sorted by similarity
        """
        results = []

        for i, doc in enumerate(corpus):
            similarity = self.compare(text, doc)
            preview = doc[:100] + "..." if len(doc) > 100 else doc
            results.append({
                "index": i,
                "similarity": similarity,
                "text": preview
            })

        # Sort by similarity descending
        results.sort(key=lambda x: x["similarity"], reverse=True)
        return results

    def similarity_matrix(self, documents: List[str],
                         labels: List[str] = None) -> pd.DataFrame:
        """
        Create pairwise similarity matrix.

        Args:
            documents: List of document texts
            labels: Optional labels for documents

        Returns:
            DataFrame with similarity matrix
        """
        n = len(documents)

        if labels is None:
            labels = [f"doc_{i}" for i in range(n)]

        # Use TF-IDF for efficiency
        vectorizer = TfidfVectorizer()
        try:
            vectors = vectorizer.fit_transform(documents)
            sim_matrix = cosine_similarity(vectors)
        except ValueError:
            sim_matrix = np.eye(n)

        return pd.DataFrame(sim_matrix, index=labels, columns=labels)

    def find_duplicates(self, documents: List[str],
                       threshold: float = 0.8) -> List[Dict]:
        """
        Find near-duplicate documents.

        Args:
            documents: List of document texts
            threshold: Minimum similarity to consider duplicate

        Returns:
            List of duplicate pairs
        """
        matrix = self.similarity_matrix(documents)
        duplicates = []

        for i in range(len(documents)):
            for j in range(i + 1, len(documents)):
                sim = matrix.iloc[i, j]
                if sim >= threshold:
                    duplicates.append({
                        "doc1_index": i,
                        "doc2_index": j,
                        "similarity": float(sim)
                    })

        # Sort by similarity descending
        duplicates.sort(key=lambda x: x["similarity"], reverse=True)
        return duplicates

    def compare_folder(self, folder: str, threshold: float = None) -> Dict:
        """
        Compare all documents in a folder.

        Args:
            folder: Folder path
            threshold: Optional threshold for similar pairs

        Returns:
            Dictionary with comparison results
        """
        supported = {'.txt', '.md', '.html', '.htm', '.doc', '.docx', '.pdf'}
        documents = {}

        for file in Path(folder).iterdir():
            if file.suffix.lower() in supported:
                try:
                    with open(file, 'r', encoding='utf-8', errors='ignore') as f:
                        documents[file.name] = f.read()
                except Exception:
                    pass

        if len(documents) < 2:
            return {"error": "Need at least 2 documents to compare"}

        files = list(documents.keys())
        texts = list(documents.values())

        matrix = self.similarity_matrix(texts, files)

        # Find similar pairs
        similar_pairs = []
        for i, file1 in enumerate(files):
            for j, file2 in enumerate(files):
                if i < j:
                    sim = matrix.iloc[i, j]
                    if threshold is None or sim >= threshold:
                        similar_pairs.append({
                            "file1": file1,
                            "file2": file2,
                            "similarity": float(sim)
                        })

        similar_pairs.sort(key=lambda x: x["similarity"], reverse=True)

        return {
            "files": files,
            "comparisons": len(similar_pairs),
            "similar_pairs": similar_pairs,
            "matrix": matrix
        }

    def find_most_similar(self, text: str, folder: str,
                         top_n: int = 5) -> List[Dict]:
        """
        Find most similar documents to query text.

        Args:
            text: Query text
            folder: Folder to search
            top_n: Number of results

        Returns:
            List of most similar documents
        """
        supported = {'.txt', '.md', '.html', '.htm'}
        documents = {}

        for file in Path(folder).iterdir():
            if file.suffix.lower() in supported:
                try:
                    with open(file, 'r', encoding='utf-8', errors='ignore') as f:
                        documents[file.name] = f.read()
                except Exception:
                    pass

        results = []
        for filename, content in documents.items():
            sim = self.compare(text, content)
            results.append({
                "file": filename,
                "similarity": sim
            })

        results.sort(key=lambda x: x["similarity"], reverse=True)
        return results[:top_n]

    def compare_with_details(self, text1: str, text2: str) -> Dict:
        """
        Compare texts with detailed analysis.

        Args:
            text1: First text
            text2: Second text

        Returns:
            Detailed comparison result
        """
        similarity = self.compare(text1, text2)

        tokens1 = set(self._tokenize(text1))
        tokens2 = set(self._tokenize(text2))

        common = tokens1 & tokens2
        unique1 = tokens1 - tokens2
        unique2 = tokens2 - tokens1

        # Interpretation
        if similarity >= 0.9:
            interpretation = "Very high similarity - likely duplicate or near-duplicate"
        elif similarity >= 0.7:
            interpretation = "High similarity - likely related or derived content"
        elif similarity >= 0.5:
            interpretation = "Moderate similarity - may share common topics"
        elif similarity >= 0.3:
            interpretation = "Low similarity - some overlap"
        else:
            interpretation = "Very low similarity - different content"

        return {
            "similarity": similarity,
            "method": self.method,
            "text1_length": len(text1),
            "text2_length": len(text2),
            "common_words": len(common),
            "unique_words_text1": len(unique1),
            "unique_words_text2": len(unique2),
            "interpretation": interpretation
        }


def main():
    parser = argparse.ArgumentParser(
        description="Content Similarity Checker - Compare documents for similarity"
    )

    parser.add_argument("--text1", help="First text to compare")
    parser.add_argument("--text2", help="Second text to compare")
    parser.add_argument("--file1", help="First file to compare")
    parser.add_argument("--file2", help="Second file to compare")
    parser.add_argument("--folder", help="Folder of documents to compare")
    parser.add_argument("--method", "-m", choices=["cosine", "jaccard", "levenshtein", "tfidf"],
                       default="cosine", help="Comparison method (default: cosine)")
    parser.add_argument("--threshold", "-t", type=float,
                       help="Similarity threshold for finding duplicates")
    parser.add_argument("--output", "-o", help="Output file for results")
    parser.add_argument("--all-methods", "-a", action="store_true",
                       help="Compare using all methods")
    parser.add_argument("--json", action="store_true", help="Output as JSON")

    args = parser.parse_args()

    checker = SimilarityChecker(method=args.method)

    if args.text1 and args.text2:
        if args.all_methods:
            result = checker.compare_all_methods(args.text1, args.text2)
            if args.json:
                print(json.dumps(result, indent=2))
            else:
                print("\n=== Similarity Comparison ===")
                for method, score in result.items():
                    print(f"{method.capitalize()}: {score:.4f}")
        else:
            result = checker.compare_with_details(args.text1, args.text2)
            if args.json:
                print(json.dumps(result, indent=2))
            else:
                print(f"\nSimilarity: {result['similarity']:.4f} ({result['similarity']:.1%})")
                print(f"Method: {result['method']}")
                print(f"Common words: {result['common_words']}")
                print(f"\n{result['interpretation']}")

    elif args.file1 and args.file2:
        with open(args.file1, 'r', encoding='utf-8', errors='ignore') as f:
            text1 = f.read()
        with open(args.file2, 'r', encoding='utf-8', errors='ignore') as f:
            text2 = f.read()

        if args.all_methods:
            result = checker.compare_all_methods(text1, text2)
        else:
            result = checker.compare_with_details(text1, text2)

        if args.json:
            print(json.dumps(result, indent=2))
        else:
            print(f"\n=== Comparing Files ===")
            print(f"File 1: {args.file1}")
            print(f"File 2: {args.file2}")
            if isinstance(result.get('similarity'), float):
                print(f"\nSimilarity: {result['similarity']:.4f} ({result['similarity']:.1%})")
                print(f"\n{result.get('interpretation', '')}")
            else:
                for method, score in result.items():
                    if isinstance(score, float):
                        print(f"{method.capitalize()}: {score:.4f}")

    elif args.folder:
        result = checker.compare_folder(args.folder, threshold=args.threshold)

        if args.json:
            # Convert matrix to dict for JSON
            if "matrix" in result:
                result["matrix"] = result["matrix"].to_dict()
            print(json.dumps(result, indent=2))
        else:
            print(f"\n=== Folder Comparison: {args.folder} ===")
            print(f"Files: {len(result.get('files', []))}")
            print(f"Comparisons: {result.get('comparisons', 0)}")

            if result.get("similar_pairs"):
                print(f"\nSimilar pairs (threshold={args.threshold or 'none'}):")
                for pair in result["similar_pairs"][:20]:
                    print(f"  {pair['file1']} <-> {pair['file2']}: {pair['similarity']:.2%}")
                if len(result["similar_pairs"]) > 20:
                    print(f"  ... and {len(result['similar_pairs']) - 20} more pairs")

        if args.output and "matrix" in result:
            matrix = result["matrix"]
            if isinstance(matrix, pd.DataFrame):
                matrix.to_csv(args.output)
                print(f"\nMatrix saved to: {args.output}")

    else:
        parser.print_help()


if __name__ == "__main__":
    main()
