"""Data analysis pattern with loading, filtering, and visualization."""

import marimo

app = marimo.App()


@app.cell
def _():
    """Imports - return all dependencies."""
    import marimo as mo
    import polars as pl

    return mo, pl


@app.cell
def _(pl):
    """Load data from file or URL."""
    # Replace with actual data source
    df = pl.read_csv("data.csv")

    # Alternative: create sample data for testing
    # df = pl.DataFrame({
    #     "date": pl.date_range(date(2024, 1, 1), date(2024, 12, 31), eager=True),
    #     "value": [random.random() * 100 for _ in range(366)],
    #     "category": [random.choice(["A", "B", "C"]) for _ in range(366)],
    # })

    return (df,)


@app.cell
def _(df, mo):
    """Data inspection - show shape and schema."""
    mo.md(
        f"""
        ## Data Overview

        - **Rows**: {df.shape[0]:,}
        - **Columns**: {df.shape[1]}
        - **Memory**: {df.estimated_size() / 1e6:.2f} MB

        ### Schema
        ```
        {df.schema}
        ```
        """
    )


@app.cell
def _(df, pl):
    """Filter and transform data."""
    # Apply filters
    filtered = df.filter(pl.col("value") > 50)

    # Add computed columns
    enhanced = filtered.with_columns(
        pl.col("value").round(2).alias("value_rounded"),
        pl.col("date").dt.month().alias("month"),
    )

    return enhanced, filtered


@app.cell
def _(enhanced, pl):
    """Aggregate by category."""
    summary = (
        enhanced.group_by("category")
        .agg(
            pl.col("value").mean().alias("avg_value"),
            pl.col("value").std().alias("std_value"),
            pl.len().alias("count"),
        )
        .sort("category")
    )

    return (summary,)


@app.cell
def _(mo, summary):
    """Display summary table."""
    mo.md("## Summary by Category")
    summary


@app.cell
def _(enhanced, mo, pl):
    """Time series visualization using altair (if available)."""
    try:
        import altair as alt

        # Prepare data for plotting
        plot_data = enhanced.select(["date", "value", "category"]).to_pandas()

        chart = (
            alt.Chart(plot_data)
            .mark_line()
            .encode(
                x="date:T",
                y="value:Q",
                color="category:N",
            )
            .properties(width=600, height=300)
        )

        mo.ui.altair_chart(chart)
    except ImportError:
        mo.md("*Install altair for visualization: `pip install altair`*")


if __name__ == "__main__":
    app.run()
