"""Interactive widgets using mo.ui for user input."""

import marimo

app = marimo.App()


@app.cell
def _():
    """Imports."""
    import marimo as mo
    import polars as pl

    return mo, pl


@app.cell
def _(mo):
    """Create interactive controls."""
    # Slider for numeric input
    threshold = mo.ui.slider(
        start=0,
        stop=100,
        step=5,
        value=50,
        label="Threshold",
    )

    # Dropdown for category selection
    category = mo.ui.dropdown(
        options=["All", "A", "B", "C"],
        value="All",
        label="Category",
    )

    # Checkbox for options
    show_details = mo.ui.checkbox(
        value=False,
        label="Show detailed statistics",
    )

    # Text input
    search = mo.ui.text(
        placeholder="Search...",
        label="Filter by name",
    )

    return category, search, show_details, threshold


@app.cell
def _(category, mo, search, show_details, threshold):
    """Display controls in a form layout."""
    mo.md("## Filters")
    mo.hstack(
        [threshold, category, show_details, search],
        justify="start",
        gap=2,
    )


@app.cell
def _(pl):
    """Sample data."""
    df = pl.DataFrame(
        {
            "name": ["Alice", "Bob", "Charlie", "Diana", "Eve"],
            "category": ["A", "B", "A", "C", "B"],
            "value": [25, 60, 45, 80, 35],
        }
    )
    return (df,)


@app.cell
def _(category, df, pl, search, threshold):
    """Filter data based on widget values."""
    filtered = df

    # Apply threshold filter
    filtered = filtered.filter(pl.col("value") >= threshold.value)

    # Apply category filter
    if category.value != "All":
        filtered = filtered.filter(pl.col("category") == category.value)

    # Apply search filter
    if search.value:
        filtered = filtered.filter(
            pl.col("name").str.to_lowercase().str.contains(search.value.lower())
        )

    return (filtered,)


@app.cell
def _(filtered, mo, show_details):
    """Display results."""
    mo.md(f"## Results ({len(filtered)} rows)")
    filtered

    if show_details.value and len(filtered) > 0:
        mo.md(
            f"""
            ### Statistics
            - **Mean value**: {filtered["value"].mean():.1f}
            - **Max value**: {filtered["value"].max()}
            - **Min value**: {filtered["value"].min()}
            """
        )


@app.cell
def _(mo):
    """Button example with state."""
    counter = mo.state(0)

    def increment():
        counter.set(counter.value + 1)

    button = mo.ui.button(
        label="Click me",
        on_click=lambda _: increment(),
    )

    return button, counter


@app.cell
def _(button, counter, mo):
    """Display button and counter."""
    mo.md("## Button Example")
    mo.hstack([button, mo.md(f"Count: **{counter.value}**")])


if __name__ == "__main__":
    app.run()
