# Marimo Debugging Workflows

## Contents

- [Quick Diagnostics](#quick-diagnostics)
- [Common Error Patterns](#common-error-patterns)
- [Runtime Debugging](#runtime-debugging)
- [Environment Issues](#environment-issues)
- [Performance Debugging](#performance-debugging)
- [Notebook Recovery](#notebook-recovery)
- [Debugging Checklist](#debugging-checklist)

## Quick Diagnostics

### Check for Errors (No Execution)
```bash
marimo check notebook.py
```
Reports syntax errors, undefined variables, circular dependencies without running cells.

### Inspect Cell Structure
```python
# get_cell_map.py - Extract cell info from notebook
import ast
from pathlib import Path

def get_cell_map(notebook_path: str) -> dict:
    """Parse marimo notebook and return cell metadata."""
    source = Path(notebook_path).read_text()
    tree = ast.parse(source)

    cells = {}
    for node in ast.walk(tree):
        if isinstance(node, ast.FunctionDef):
            # Check for @app.cell decorator
            for decorator in node.decorator_list:
                if hasattr(decorator, 'attr') and decorator.attr == 'cell':
                    cells[node.name] = {
                        'lineno': node.lineno,
                        'args': [arg.arg for arg in node.args.args],
                        'returns': _extract_returns(node),
                    }
    return cells

def _extract_returns(func_node) -> list:
    """Extract return variable names from function."""
    returns = []
    for node in ast.walk(func_node):
        if isinstance(node, ast.Return) and node.value:
            if isinstance(node.value, ast.Tuple):
                returns = [elt.id for elt in node.value.elts
                          if isinstance(elt, ast.Name)]
            elif isinstance(node.value, ast.Name):
                returns = [node.value.id]
    return returns
```

## Common Error Patterns

### Variable Redefinition
**Error**: `NameError: name 'x' is defined in multiple cells`

**Debug**:
```bash
grep -n "^    x = " notebook.py
```

**Fix**: Rename one variable or merge cells.

### Circular Dependency
**Error**: `CircularDependencyError`

**Debug**:
1. Open in marimo editor: `marimo edit notebook.py`
2. View → Dependency graph
3. Look for cycles (highlighted in red)

**Fix**: Break cycle by:
- Merging dependent cells
- Extracting shared logic to a function
- Restructuring data flow

### Missing Return
**Symptom**: Variable undefined in downstream cell

**Debug**:
```python
# Check if cell returns the variable
@app.cell
def _():
    df = load_data()
    # Missing: return df,
```

**Fix**: Add return statement with trailing comma.

### Import Not Available
**Error**: `NameError: name 'pl' is not defined`

**Debug**: Check import cell exists and returns the module:
```python
@app.cell
def _():
    import polars as pl
    return pl,  # Must return!
```

## Runtime Debugging

### Inspect Cell Output
```python
@app.cell
def _(df):
    # Debug: inspect intermediate state
    print(f"Shape: {df.shape}")
    print(f"Columns: {df.columns}")
    print(f"Nulls:\n{df.null_count()}")

    result = df.filter(...)
    print(f"After filter: {result.shape}")
    return result,
```

### Check Data Types
```python
@app.cell
def _(df):
    mo.md(f"""
    ## Data Inspection
    - **Shape**: {df.shape}
    - **Schema**: {df.schema}
    - **Memory**: {df.estimated_size() / 1e6:.2f} MB
    """)
```

### Trace Execution Order
```python
@app.cell
def _():
    import datetime
    print(f"[{datetime.datetime.now()}] Cell A executed")
    # ... cell code
```

## Environment Issues

### Check marimo Version
```bash
marimo --version
pip show marimo
```

### Verify Dependencies
```bash
# Check if required packages are installed
python -c "import polars; print(polars.__version__)"
python -c "import marimo; print(marimo.__version__)"
```

### Virtual Environment
```bash
# Ensure correct env is active
which python
pip list | grep marimo
```

## Performance Debugging

### Profile Cell Execution
```python
@app.cell
def _(df):
    import time
    start = time.perf_counter()

    result = expensive_operation(df)

    elapsed = time.perf_counter() - start
    print(f"Execution time: {elapsed:.2f}s")
    return result,
```

### Memory Usage
```python
@app.cell
def _(df):
    import sys

    size_mb = sys.getsizeof(df) / 1e6
    print(f"DataFrame size: {size_mb:.2f} MB")

    # For polars
    if hasattr(df, 'estimated_size'):
        print(f"Polars estimate: {df.estimated_size() / 1e6:.2f} MB")
```

## Notebook Recovery

### Extract Code from Corrupted Notebook
```bash
# Marimo notebooks are valid Python, so:
python notebook.py  # Should at least parse

# Or extract cell contents:
grep -A 20 "@app.cell" notebook.py
```

### Reset Notebook State
```bash
# Clear all outputs and restart
marimo edit notebook.py --fresh
```

## Debugging Checklist

1. [ ] Run `marimo check notebook.py` for static errors
2. [ ] Check variable definitions (single source per variable)
3. [ ] Verify all cells have proper return statements
4. [ ] Check for circular dependencies in graph view
5. [ ] Verify imports are returned from their cells
6. [ ] Check data types match expectations
7. [ ] Verify environment has correct packages
