# Marimo Reactivity Model

## Contents

- [DAG Execution](#dag-execution)
- [Variable Rules](#variable-rules)
- [Dependency Detection](#dependency-detection)
- [Circular Dependencies](#circular-dependencies)
- [Stale Cells](#stale-cells)
- [Pure Functions Pattern](#pure-functions-pattern)
- [Global State Warning](#global-state-warning)
- [Debugging DAG Issues](#debugging-dag-issues)

## DAG Execution

Marimo builds a directed acyclic graph (DAG) from cell dependencies:

```
Cell A (defines x) → Cell B (uses x, defines y) → Cell C (uses y)
```

When Cell A changes, cells B and C automatically re-execute in order.

## Variable Rules

### Single Definition Rule
Each variable can only be defined in ONE cell:

```python
# WRONG: x defined in multiple cells
@app.cell
def cell1():
    x = 1
    return x,

@app.cell
def cell2():
    x = 2  # ERROR: x already defined
    return x,
```

```python
# CORRECT: different names or single source
@app.cell
def cell1():
    x = 1
    return x,

@app.cell
def cell2(x):
    y = x + 1  # Uses x from cell1
    return y,
```

### Return Statement Requirements

Cells must return variables they define for other cells to use:

```python
@app.cell
def _(pl):
    df = pl.read_csv("data.csv")
    return df,  # Trailing comma required for single return

@app.cell
def _(df, pl):
    summary = df.describe()
    filtered = df.filter(pl.col("value") > 0)
    return summary, filtered  # Multiple returns
```

## Dependency Detection

Marimo automatically detects dependencies through:
1. Function parameters (inputs from other cells)
2. Return values (outputs to other cells)

```python
@app.cell
def _(pl):  # Depends on polars import
    df = pl.DataFrame({"a": [1, 2, 3]})
    return df,

@app.cell
def _(df):  # Depends on df from above cell
    print(df.shape)
```

## Circular Dependencies

Circular dependencies cause errors:

```
Cell A uses y from Cell B
Cell B uses x from Cell A
```

**Fix**: Merge cells or restructure to break the cycle.

## Stale Cells

A cell becomes "stale" when:
1. Its dependencies have changed
2. It hasn't re-executed yet

In interactive mode, marimo highlights stale cells. Run them to update.

## Pure Functions Pattern

For complex logic, use pure functions within cells:

```python
@app.cell
def _():
    def process_data(df):
        """Pure function - no side effects."""
        return df.filter(pl.col("valid") == True)
    return process_data,

@app.cell
def _(df, process_data):
    clean_df = process_data(df)
    return clean_df,
```

## Global State Warning

Avoid global mutable state:

```python
# BAD: Mutable global affects reactivity
results = []

@app.cell
def _():
    results.append(1)  # Side effect, breaks reactivity

# GOOD: Return new values
@app.cell
def _():
    results = [1]
    return results,
```

## Debugging DAG Issues

Use marimo's built-in graph view:
- `marimo edit notebook.py` → View → Dependency graph
- Or programmatically: `app.graph()` in a cell

Check for:
- Unexpected dependencies
- Missing connections
- Isolated cells (no inputs/outputs)
