# Marimo SQL Integration

## Contents

- [SQL Cells](#sql-cells)
- [Basic SQL on DataFrames](#basic-sql-on-dataframes)
- [SQL Cell Syntax](#sql-cell-syntax)
- [Database Connections](#database-connections)
- [Parameterized Queries](#parameterized-queries)
- [Common Query Patterns](#common-query-patterns)
- [Interactive SQL](#interactive-sql)
- [Performance Tips](#performance-tips)
- [Troubleshooting](#troubleshooting)

## SQL Cells

Marimo provides native SQL support through `mo.sql()` for querying DataFrames and databases.

## Basic SQL on DataFrames

```python
@app.cell
def _(mo, pl):
    # Create sample data
    df = pl.DataFrame({
        "id": [1, 2, 3, 4, 5],
        "name": ["Alice", "Bob", "Charlie", "Diana", "Eve"],
        "value": [100, 200, 150, 300, 250],
        "category": ["A", "B", "A", "B", "A"],
    })
    return df,

@app.cell
def _(df, mo):
    # Query DataFrame with SQL
    result = mo.sql(
        """
        SELECT category, AVG(value) as avg_value
        FROM df
        GROUP BY category
        ORDER BY avg_value DESC
        """,
        dataframes={"df": df}
    )
    return result,
```

## SQL Cell Syntax

### Implicit DataFrame Binding

```python
@app.cell
def _(df, mo):
    # df is automatically available if in scope
    result = mo.sql(
        """
        SELECT * FROM df WHERE value > 100
        """
    )
    return result,
```

### Multiple DataFrames

```python
@app.cell
def _(customers, orders, mo):
    result = mo.sql(
        """
        SELECT c.name, o.total
        FROM customers c
        JOIN orders o ON c.id = o.customer_id
        WHERE o.total > 500
        """,
        dataframes={
            "customers": customers,
            "orders": orders,
        }
    )
    return result,
```

## Database Connections

### SQLite

```python
@app.cell
def _(mo):
    import sqlite3

    conn = sqlite3.connect("database.db")
    return conn,

@app.cell
def _(conn, mo):
    result = mo.sql(
        """
        SELECT * FROM users WHERE active = 1
        """,
        connection=conn
    )
    return result,
```

### PostgreSQL with psycopg2

```python
@app.cell
def _(mo):
    import psycopg2

    conn = psycopg2.connect(
        host="localhost",
        database="mydb",
        user="user",
        password="password"
    )
    return conn,

@app.cell
def _(conn, mo):
    result = mo.sql(
        """
        SELECT * FROM orders
        WHERE created_at > '2024-01-01'
        LIMIT 100
        """,
        connection=conn
    )
    return result,
```

### DuckDB (Recommended for Analytics)

```python
@app.cell
def _():
    import duckdb

    # DuckDB works great with polars and parquet
    conn = duckdb.connect()
    return conn,

@app.cell
def _(conn, mo):
    # Query parquet files directly
    result = mo.sql(
        """
        SELECT * FROM 'data/*.parquet'
        WHERE date >= '2024-01-01'
        """,
        connection=conn
    )
    return result,
```

## Parameterized Queries

### Using f-strings (Simple Cases)

```python
@app.cell
def _(df, mo, threshold):
    # threshold is a marimo widget
    result = mo.sql(
        f"""
        SELECT * FROM df
        WHERE value > {threshold.value}
        """
    )
    return result,
```

### Using Parameters (Safer)

```python
@app.cell
def _(conn, mo, user_input):
    # Prevents SQL injection
    result = mo.sql(
        """
        SELECT * FROM users
        WHERE name = :name
        """,
        connection=conn,
        params={"name": user_input.value}
    )
    return result,
```

## Common Query Patterns

### Aggregations

```python
@app.cell
def _(df, mo):
    summary = mo.sql(
        """
        SELECT
            category,
            COUNT(*) as count,
            SUM(value) as total,
            AVG(value) as average,
            MIN(value) as min_val,
            MAX(value) as max_val
        FROM df
        GROUP BY category
        """
    )
    return summary,
```

### Window Functions

```python
@app.cell
def _(df, mo):
    ranked = mo.sql(
        """
        SELECT
            *,
            ROW_NUMBER() OVER (PARTITION BY category ORDER BY value DESC) as rank,
            SUM(value) OVER (PARTITION BY category) as category_total
        FROM df
        """
    )
    return ranked,
```

### CTEs (Common Table Expressions)

```python
@app.cell
def _(df, mo):
    result = mo.sql(
        """
        WITH category_stats AS (
            SELECT category, AVG(value) as avg_value
            FROM df
            GROUP BY category
        ),
        ranked AS (
            SELECT *, ROW_NUMBER() OVER (ORDER BY avg_value DESC) as rank
            FROM category_stats
        )
        SELECT * FROM ranked WHERE rank <= 3
        """
    )
    return result,
```

### Joins

```python
@app.cell
def _(customers, orders, mo):
    joined = mo.sql(
        """
        SELECT
            c.id,
            c.name,
            COUNT(o.id) as order_count,
            COALESCE(SUM(o.total), 0) as total_spent
        FROM customers c
        LEFT JOIN orders o ON c.id = o.customer_id
        GROUP BY c.id, c.name
        ORDER BY total_spent DESC
        """
    )
    return joined,
```

## Interactive SQL

### Combine SQL with Widgets

```python
@app.cell
def _(mo):
    category_filter = mo.ui.dropdown(
        options=["All", "A", "B", "C"],
        value="All",
        label="Category",
    )
    limit_slider = mo.ui.slider(
        start=10, stop=100, step=10, value=50,
        label="Limit",
    )
    return category_filter, limit_slider,

@app.cell
def _(category_filter, df, limit_slider, mo):
    where_clause = ""
    if category_filter.value != "All":
        where_clause = f"WHERE category = '{category_filter.value}'"

    result = mo.sql(
        f"""
        SELECT * FROM df
        {where_clause}
        ORDER BY value DESC
        LIMIT {limit_slider.value}
        """
    )
    return result,
```

### Display SQL Results

```python
@app.cell
def _(mo, result):
    mo.md(f"## Query Results ({len(result)} rows)")
    mo.ui.table(result, pagination=True, page_size=20)
```

## Performance Tips

### Use LIMIT for Large Datasets

```python
@app.cell
def _(conn, mo):
    # Always limit results when exploring
    sample = mo.sql(
        """
        SELECT * FROM large_table
        ORDER BY RANDOM()
        LIMIT 1000
        """,
        connection=conn
    )
    return sample,
```

### Materialize Intermediate Results

```python
@app.cell
def _(conn, mo):
    # Create temp table for repeated use
    mo.sql(
        """
        CREATE TEMP TABLE IF NOT EXISTS filtered_data AS
        SELECT * FROM raw_data WHERE valid = true
        """,
        connection=conn
    )

@app.cell
def _(conn, mo):
    # Use temp table in multiple queries
    result = mo.sql(
        """
        SELECT * FROM filtered_data
        WHERE value > 100
        """,
        connection=conn
    )
    return result,
```

### Index Awareness

```sql
-- When working with databases, check indexes
EXPLAIN ANALYZE SELECT * FROM users WHERE email = 'test@example.com';

-- Add indexes for frequently filtered columns
CREATE INDEX IF NOT EXISTS idx_users_email ON users(email);
```

## Troubleshooting

### Common Errors

| Error | Cause | Fix |
|-------|-------|-----|
| `Table not found` | DataFrame not in scope | Pass via `dataframes={}` |
| `Column not found` | Typo in column name | Check `df.columns` |
| `Type mismatch` | Comparing incompatible types | Cast explicitly |
| `Connection closed` | Database connection dropped | Reconnect in separate cell |

### Debugging Queries

```python
@app.cell
def _(df, mo):
    # Print the query for debugging
    query = """
        SELECT * FROM df WHERE value > 100
    """
    print(f"Executing: {query}")

    result = mo.sql(query)
    print(f"Returned {len(result)} rows")
    return result,
```
