# Marimo Interactive Widgets

## Contents

- [mo.ui Overview](#moui-overview)
- [Input Widgets](#input-widgets)
- [Buttons and Actions](#buttons-and-actions)
- [Layout Components](#layout-components)
- [Data Display](#data-display)
- [Visualization](#visualization)
- [State Management](#state-management)
- [Common Patterns](#common-patterns)

## mo.ui Overview

Marimo provides interactive UI components through `mo.ui`. Widgets automatically trigger cell re-execution when values change.

## Input Widgets

### Slider

```python
@app.cell
def _(mo):
    threshold = mo.ui.slider(
        start=0,
        stop=100,
        step=5,
        value=50,
        label="Threshold",
        show_value=True,
    )
    return threshold,

@app.cell
def _(threshold):
    # Access value with .value
    print(f"Current threshold: {threshold.value}")
```

### Dropdown

```python
@app.cell
def _(mo):
    category = mo.ui.dropdown(
        options=["All", "A", "B", "C"],
        value="All",
        label="Category",
    )
    return category,
```

### Multi-select

```python
@app.cell
def _(mo):
    selected = mo.ui.multiselect(
        options=["Option A", "Option B", "Option C"],
        value=["Option A"],
        label="Select items",
    )
    return selected,
```

### Text Input

```python
@app.cell
def _(mo):
    search = mo.ui.text(
        placeholder="Enter search term...",
        label="Search",
        value="",
    )
    return search,
```

### Text Area

```python
@app.cell
def _(mo):
    notes = mo.ui.text_area(
        placeholder="Enter notes...",
        label="Notes",
        rows=5,
    )
    return notes,
```

### Checkbox

```python
@app.cell
def _(mo):
    enabled = mo.ui.checkbox(
        value=False,
        label="Enable feature",
    )
    return enabled,
```

### Radio Buttons

```python
@app.cell
def _(mo):
    choice = mo.ui.radio(
        options=["Option A", "Option B", "Option C"],
        value="Option A",
        label="Select one",
    )
    return choice,
```

### Date Picker

```python
@app.cell
def _(mo):
    from datetime import date

    selected_date = mo.ui.date(
        value=date.today(),
        label="Select date",
    )
    return selected_date,
```

### Date Range

```python
@app.cell
def _(mo):
    from datetime import date

    date_range = mo.ui.date_range(
        start=date(2024, 1, 1),
        stop=date(2024, 12, 31),
        label="Date range",
    )
    return date_range,
```

## Buttons and Actions

### Basic Button

```python
@app.cell
def _(mo):
    button = mo.ui.button(
        label="Click me",
        kind="primary",  # primary, secondary, danger, warn
    )
    return button,

@app.cell
def _(button):
    # Button value increments on each click
    print(f"Button clicked {button.value} times")
```

### Button with Callback

```python
@app.cell
def _(mo):
    counter = mo.state(0)

    def on_click(_):
        counter.set(counter.value + 1)

    button = mo.ui.button(
        label="Increment",
        on_click=on_click,
    )
    return button, counter,

@app.cell
def _(counter, mo):
    mo.md(f"Count: **{counter.value}**")
```

### Run Button

```python
@app.cell
def _(mo):
    # Only runs when button is clicked
    run = mo.ui.run_button(label="Run Analysis")
    return run,

@app.cell
def _(run):
    if run.value:
        # Expensive computation here
        result = perform_analysis()
```

## Layout Components

### Horizontal Stack

```python
@app.cell
def _(mo, slider1, slider2, slider3):
    mo.hstack(
        [slider1, slider2, slider3],
        justify="start",  # start, center, end, space-between
        gap=2,
    )
```

### Vertical Stack

```python
@app.cell
def _(mo, widget1, widget2, widget3):
    mo.vstack(
        [widget1, widget2, widget3],
        align="stretch",
        gap=1,
    )
```

### Accordion

```python
@app.cell
def _(mo):
    mo.accordion({
        "Section 1": mo.md("Content for section 1"),
        "Section 2": mo.md("Content for section 2"),
        "Section 3": mo.md("Content for section 3"),
    })
```

### Tabs

```python
@app.cell
def _(mo):
    mo.ui.tabs({
        "Data": data_table,
        "Chart": chart,
        "Summary": summary,
    })
```

## Data Display

### DataTable

```python
@app.cell
def _(df, mo):
    mo.ui.table(
        df,
        selection="multi",  # none, single, multi
        pagination=True,
        page_size=10,
    )
```

### Interactive DataFrame

```python
@app.cell
def _(df, mo):
    # Returns selected rows
    table = mo.ui.dataframe(df)
    return table,

@app.cell
def _(table):
    # Access selected data
    selected = table.value
```

## Visualization

### Altair Chart

```python
@app.cell
def _(mo):
    import altair as alt

    chart = alt.Chart(data).mark_bar().encode(
        x='category',
        y='value',
    )

    # Interactive chart with selection
    mo.ui.altair_chart(chart)
```

### Plotly Chart

```python
@app.cell
def _(mo):
    import plotly.express as px

    fig = px.scatter(df, x='x', y='y', color='category')
    mo.ui.plotly(fig)
```

## State Management

### mo.state for Persistent Values

```python
@app.cell
def _(mo):
    # State persists across cell re-runs
    counter = mo.state(0)
    items = mo.state([])
    return counter, items,

@app.cell
def _(counter, items, mo):
    def add_item():
        items.set(items.value + [f"Item {len(items.value) + 1}"])
        counter.set(counter.value + 1)

    button = mo.ui.button(label="Add Item", on_click=lambda _: add_item())
    return button,
```

### Batch Updates

```python
@app.cell
def _(mo):
    state1 = mo.state(0)
    state2 = mo.state(0)

    def update_both():
        # Batch updates to avoid multiple re-renders
        with mo.batch():
            state1.set(state1.value + 1)
            state2.set(state2.value + 1)
```

## Common Patterns

### Form with Multiple Inputs

```python
@app.cell
def _(mo):
    form = mo.ui.form(
        mo.vstack([
            mo.ui.text(label="Name", placeholder="Enter name"),
            mo.ui.number(label="Age", start=0, stop=120),
            mo.ui.dropdown(options=["A", "B", "C"], label="Category"),
        ]),
        submit_button_label="Submit",
    )
    return form,

@app.cell
def _(form):
    if form.value:
        name, age, category = form.value
        process_form(name, age, category)
```

### Conditional Widget Display

```python
@app.cell
def _(advanced_mode, mo):
    if advanced_mode.value:
        mo.vstack([
            advanced_slider,
            advanced_dropdown,
            advanced_checkbox,
        ])
    else:
        mo.md("Enable advanced mode for more options")
```

### Dynamic Options

```python
@app.cell
def _(category, mo):
    # Options depend on category selection
    if category.value == "A":
        options = ["A1", "A2", "A3"]
    elif category.value == "B":
        options = ["B1", "B2"]
    else:
        options = ["C1"]

    sub_category = mo.ui.dropdown(
        options=options,
        label="Sub-category",
    )
    return sub_category,
```
