#!/usr/bin/env bash
# Quick validation wrapper for marimo notebooks
#
# Usage:
#   check_notebook.sh notebook.py
#   check_notebook.sh *.py
#
# Runs:
#   1. Python syntax check (fast)
#   2. marimo check (static analysis)
#   3. Cell map extraction (structure overview)

set -euo pipefail

if [[ $# -eq 0 ]]; then
    echo "Usage: check_notebook.sh <notebook.py> [notebook2.py ...]"
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

check_notebook() {
    local notebook="$1"
    echo "=== Checking: $notebook ==="
    echo

    # 1. Python syntax check
    echo "1. Syntax check..."
    if python3 -m py_compile "$notebook" 2>&1; then
        echo "   OK"
    else
        echo "   FAILED"
        return 1
    fi

    # 2. marimo static analysis
    echo "2. Marimo check..."
    if command -v marimo &>/dev/null; then
        if marimo check "$notebook" 2>&1; then
            echo "   OK"
        else
            echo "   FAILED (see errors above)"
            return 1
        fi
    else
        echo "   SKIPPED (marimo not installed)"
    fi

    # 3. Cell structure
    echo "3. Cell structure..."
    if [[ -f "$SCRIPT_DIR/get_cell_map.py" ]]; then
        python3 "$SCRIPT_DIR/get_cell_map.py" "$notebook"
    else
        echo "   SKIPPED (get_cell_map.py not found)"
    fi

    echo
}

exit_code=0

for notebook in "$@"; do
    if [[ -f "$notebook" ]]; then
        if ! check_notebook "$notebook"; then
            exit_code=1
        fi
    else
        echo "Warning: File not found: $notebook"
        exit_code=1
    fi
done

exit $exit_code
