#!/usr/bin/env python3
"""Extract cell metadata from a marimo notebook without executing it.

Usage:
    python get_cell_map.py notebook.py
    python get_cell_map.py notebook.py --json

Output includes:
    - Cell function names
    - Line numbers
    - Input dependencies (function parameters)
    - Output variables (return values)
"""

import ast
import json
import sys
from pathlib import Path
from typing import Any


def extract_returns(func_node: ast.FunctionDef) -> list[str]:
    """Extract return variable names from function."""
    returns = []
    for node in ast.walk(func_node):
        if isinstance(node, ast.Return) and node.value:
            if isinstance(node.value, ast.Tuple):
                returns = [
                    elt.id
                    for elt in node.value.elts
                    if isinstance(elt, ast.Name)
                ]
            elif isinstance(node.value, ast.Name):
                returns = [node.value.id]
    return returns


def get_cell_map(notebook_path: str) -> dict[str, dict[str, Any]]:
    """Parse marimo notebook and return cell metadata.

    Args:
        notebook_path: Path to .py file containing marimo notebook

    Returns:
        Dictionary mapping cell function names to their metadata:
        - lineno: Line number where cell is defined
        - inputs: List of input dependencies (function parameters)
        - outputs: List of output variables (return values)
    """
    source = Path(notebook_path).read_text()
    tree = ast.parse(source)

    cells = {}
    for node in ast.walk(tree):
        if isinstance(node, ast.FunctionDef):
            # Check for @app.cell decorator
            for decorator in node.decorator_list:
                if hasattr(decorator, "attr") and decorator.attr == "cell":
                    cells[node.name] = {
                        "lineno": node.lineno,
                        "inputs": [arg.arg for arg in node.args.args],
                        "outputs": extract_returns(node),
                    }
    return cells


def print_cell_map(cells: dict[str, dict[str, Any]], as_json: bool = False) -> None:
    """Print cell map in human-readable or JSON format."""
    if as_json:
        print(json.dumps(cells, indent=2))
        return

    print(f"Found {len(cells)} cells:\n")
    for name, info in sorted(cells.items(), key=lambda x: x[1]["lineno"]):
        inputs = ", ".join(info["inputs"]) if info["inputs"] else "(none)"
        outputs = ", ".join(info["outputs"]) if info["outputs"] else "(none)"
        print(f"  {name} (line {info['lineno']})")
        print(f"    inputs:  {inputs}")
        print(f"    outputs: {outputs}")
        print()


def main():
    if len(sys.argv) < 2:
        print("Usage: python get_cell_map.py <notebook.py> [--json]")
        sys.exit(1)

    notebook_path = sys.argv[1]
    as_json = "--json" in sys.argv

    if not Path(notebook_path).exists():
        print(f"Error: File not found: {notebook_path}")
        sys.exit(1)

    try:
        cells = get_cell_map(notebook_path)
        print_cell_map(cells, as_json=as_json)
    except SyntaxError as e:
        print(f"Syntax error in notebook: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
