#!/usr/bin/env python3
"""
Helper script for interacting with the Shownotes.io API
"""

import requests
import sys
import json
from urllib.parse import urlencode

BASE_URL = "https://plugins.shownotes.io"

def search_apple_podcasts(query, token=None):
    """
    Search for podcast episodes on Apple Podcasts
    
    Args:
        query: Search query string
        token: OAuth2 access token (optional for now)
    
    Returns:
        List of episode results
    """
    url = f"{BASE_URL}/apple.php"
    params = {"q": query}
    
    headers = {}
    if token:
        headers["Authorization"] = f"Bearer {token}"
    
    try:
        response = requests.get(url, params=params, headers=headers)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error searching Apple Podcasts: {e}", file=sys.stderr)
        return None

def transcribe_audio(enclosure, podcast, title, token=None):
    """
    Request transcription of a podcast episode
    
    Args:
        enclosure: URL to the audio file
        podcast: Podcast name
        title: Episode title
        token: OAuth2 access token (optional for now)
    
    Returns:
        Dict with audiolink where transcript will appear
    """
    url = f"{BASE_URL}/airplugin.php"
    params = {
        "enclosure": enclosure,
        "podcast": podcast,
        "title": title
    }
    
    headers = {}
    if token:
        headers["Authorization"] = f"Bearer {token}"
    
    try:
        response = requests.get(url, params=params, headers=headers)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error requesting transcription: {e}", file=sys.stderr)
        return None

def get_shownotes_transcript(show, token=None):
    """
    Get existing transcript from shownotes.io database
    
    Args:
        show: Show name
        token: OAuth2 access token (optional for now)
    
    Returns:
        Dict with transcript chunks
    """
    url = f"{BASE_URL}/captions-shownotes.php"
    params = {"show": show}
    
    headers = {}
    if token:
        headers["Authorization"] = f"Bearer {token}"
    
    try:
        response = requests.get(url, params=params, headers=headers)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error getting transcript: {e}", file=sys.stderr)
        return None

def search_youtube(show, token=None):
    """
    Search YouTube for videos matching show name
    
    Args:
        show: Show name
        token: OAuth2 access token (optional for now)
    
    Returns:
        List of video results
    """
    url = f"{BASE_URL}/search-youtube.php"
    params = {"show": show}
    
    headers = {}
    if token:
        headers["Authorization"] = f"Bearer {token}"
    
    try:
        response = requests.get(url, params=params, headers=headers)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error searching YouTube: {e}", file=sys.stderr)
        return None

def get_youtube_transcript(videoid, token=None):
    """
    Get transcript from YouTube video
    
    Args:
        videoid: 11-character YouTube video ID
        token: OAuth2 access token (optional for now)
    
    Returns:
        Dict with transcript chunks
    """
    if len(videoid) != 11:
        print(f"Error: Video ID must be exactly 11 characters, got {len(videoid)}", file=sys.stderr)
        return None
    
    url = f"{BASE_URL}/captions-youtube.php"
    params = {"videoid": videoid}
    
    headers = {}
    if token:
        headers["Authorization"] = f"Bearer {token}"
    
    try:
        response = requests.get(url, params=params, headers=headers)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error getting YouTube transcript: {e}", file=sys.stderr)
        return None

def combine_transcript_chunks(transcript_data):
    """
    Combine transcript chunks into single text
    
    Args:
        transcript_data: Dict with 'transcript' key containing chunk1 and chunk2
    
    Returns:
        Combined transcript text
    """
    if not transcript_data or 'transcript' not in transcript_data:
        return ""
    
    transcript = transcript_data['transcript']
    chunk1 = transcript.get('chunk1', '')
    chunk2 = transcript.get('chunk2', '')
    
    return chunk1 + chunk2

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage:")
        print("  python shownotes_api.py search-apple <query>")
        print("  python shownotes_api.py transcribe <enclosure> <podcast> <title>")
        print("  python shownotes_api.py get-transcript <show>")
        print("  python shownotes_api.py search-youtube <show>")
        print("  python shownotes_api.py get-youtube <videoid>")
        sys.exit(1)
    
    command = sys.argv[1]
    
    if command == "search-apple" and len(sys.argv) >= 3:
        query = " ".join(sys.argv[2:])
        results = search_apple_podcasts(query)
        if results:
            print(json.dumps(results, indent=2))
    
    elif command == "transcribe" and len(sys.argv) >= 5:
        enclosure = sys.argv[2]
        podcast = sys.argv[3]
        title = " ".join(sys.argv[4:])
        result = transcribe_audio(enclosure, podcast, title)
        if result:
            print(json.dumps(result, indent=2))
    
    elif command == "get-transcript" and len(sys.argv) >= 3:
        show = " ".join(sys.argv[2:])
        result = get_shownotes_transcript(show)
        if result:
            print(json.dumps(result, indent=2))
    
    elif command == "search-youtube" and len(sys.argv) >= 3:
        show = " ".join(sys.argv[2:])
        results = search_youtube(show)
        if results:
            print(json.dumps(results, indent=2))
    
    elif command == "get-youtube" and len(sys.argv) >= 3:
        videoid = sys.argv[2]
        result = get_youtube_transcript(videoid)
        if result:
            full_transcript = combine_transcript_chunks(result)
            print(full_transcript)
    
    else:
        print("Invalid command or arguments", file=sys.stderr)
        sys.exit(1)
