#!/usr/bin/env python3
"""Compile outline project into final documents."""

import json
import os
import sys
from datetime import datetime


def find_state_file(path: str) -> str:
    """Find the outline state file, checking new location first then legacy."""
    # New location: .crucible/state/outline-state.json
    new_path = os.path.join(path, ".crucible", "state", "outline-state.json")
    if os.path.exists(new_path):
        return new_path

    # Legacy location: state.json at project root
    legacy_path = os.path.join(path, "state.json")
    if os.path.exists(legacy_path):
        return legacy_path

    return None


def compile_outline(path: str) -> list:
    """
    Compile outline project into markdown documents.

    Args:
        path: Project directory path

    Returns:
        List of generated file paths
    """
    state_path = find_state_file(path)
    if not state_path:
        raise FileNotFoundError(
            f"No state file found. Checked:\n"
            f"  - {os.path.join(path, '.crucible', 'state', 'outline-state.json')}\n"
            f"  - {os.path.join(path, 'state.json')}"
        )

    outline_dir = os.path.join(path, "outline")
    chapter_dir = os.path.join(outline_dir, "by-chapter")

    # Load state
    with open(state_path, "r") as f:
        state = json.load(f)
    
    os.makedirs(chapter_dir, exist_ok=True)
    
    generated_files = []
    
    # Generate master outline
    master_path = os.path.join(outline_dir, "master-outline.md")
    generate_master_outline(state, master_path)
    generated_files.append(master_path)
    print(f"   [OK] Master Outline")

    # Generate chapter summaries
    summary_path = os.path.join(outline_dir, "chapter-summaries.md")
    generate_chapter_summaries(state, summary_path)
    generated_files.append(summary_path)
    print(f"   [OK] Chapter Summaries")

    # Generate scene breakdown
    scene_path = os.path.join(outline_dir, "scene-breakdown.md")
    generate_scene_breakdown(state, scene_path)
    generated_files.append(scene_path)
    print(f"   [OK] Scene Breakdown")

    # Generate foreshadowing tracker
    foreshadow_path = os.path.join(outline_dir, "foreshadowing-tracker.md")
    generate_foreshadowing_tracker(state, foreshadow_path)
    generated_files.append(foreshadow_path)
    print(f"   [OK] Foreshadowing Tracker")

    # Generate character threads
    character_path = os.path.join(outline_dir, "character-threads.md")
    generate_character_threads(state, character_path)
    generated_files.append(character_path)
    print(f"   [OK] Character Threads")
    
    # Generate individual chapter files
    for chapter in state.get("chapters", []):
        ch_num = chapter.get("number", 0)
        ch_path = os.path.join(chapter_dir, f"ch{ch_num:02d}-outline.md")
        generate_chapter_file(chapter, state, ch_path)
        generated_files.append(ch_path)
    
    if state.get("chapters"):
        print(f"   [OK] {len(state['chapters'])} Chapter Files")
    
    return generated_files


def generate_master_outline(state: dict, filepath: str):
    """Generate the master outline document."""
    content = f"""# {state.get('title', 'Untitled')} — Master Outline

## Overview
- **Book:** {state.get('book_info', 'Standalone')}
- **Chapter Count:** {state.get('structure', {}).get('chapter_count', 'TBD')}
- **Generated:** {datetime.now().strftime('%Y-%m-%d %H:%M')}

## Theme
{state.get('crucible_elements', {}).get('theme', '*[Theme from Crucible Thesis]*')}

## Core Elements
- **Burden:** {state.get('crucible_elements', {}).get('burden', '*[From Crucible Thesis]*')}
- **Fire:** {state.get('crucible_elements', {}).get('fire', '*[From Crucible Thesis]*')}
- **Core Bond:** {state.get('crucible_elements', {}).get('core_bond', '*[From Crucible Thesis]*')}
- **Dark Mirror:** {state.get('crucible_elements', {}).get('dark_mirror', '*[From Crucible Thesis]*')}

---

## Movement Breakdown

"""
    
    movements = state.get('structure', {}).get('movements', {})
    for movement_name, movement_data in movements.items():
        content += f"""### {movement_name}
{movement_data.get('summary', '*[Movement summary]*')}

**Chapters:** {movement_data.get('start_chapter', '?')}-{movement_data.get('end_chapter', '?')}

"""
    
    content += """---

## Chapter Outlines

"""
    
    for chapter in state.get('chapters', []):
        content += f"""### Chapter {chapter.get('number', '?')}: {chapter.get('title', 'Untitled')}

**Beats:** {chapter.get('beats', 'TBD')} | **Strand:** {chapter.get('strand_focus', 'TBD')} | **POV:** {chapter.get('pov', 'TBD')}

**Summary:** {chapter.get('summary', '*[Chapter summary]*')}

**Scenes:**
"""
        for i, scene in enumerate(chapter.get('scenes', []), 1):
            content += f"- Scene {i}: {scene.get('title', 'Untitled')} — {scene.get('goal', 'TBD')}\n"
        
        content += f"""
**Chapter Turn:** {chapter.get('chapter_turn', '*[How chapter ends]*')}

---

"""
    
    with open(filepath, 'w') as f:
        f.write(content)


def generate_chapter_summaries(state: dict, filepath: str):
    """Generate quick-reference chapter summaries."""
    content = f"""# Chapter Summaries — {state.get('title', 'Untitled')}

| Ch | Title | Beats | POV | Strand | Summary |
|----|-------|-------|-----|--------|---------|
"""
    
    for chapter in state.get('chapters', []):
        content += f"| {chapter.get('number', '?')} | {chapter.get('title', 'Untitled')} | {chapter.get('beats', '?')} | {chapter.get('pov', '?')} | {chapter.get('strand_focus', '?')} | {chapter.get('one_line', '*TBD*')} |\n"
    
    with open(filepath, 'w') as f:
        f.write(content)


def generate_scene_breakdown(state: dict, filepath: str):
    """Generate detailed scene list."""
    content = f"""# Scene Breakdown — {state.get('title', 'Untitled')}

"""
    
    for chapter in state.get('chapters', []):
        content += f"""## Chapter {chapter.get('number', '?')}: {chapter.get('title', 'Untitled')}

"""
        for i, scene in enumerate(chapter.get('scenes', []), 1):
            content += f"""### Scene {chapter.get('number', '?')}.{i}: {scene.get('title', 'Untitled')}

- **POV:** {scene.get('pov', chapter.get('pov', 'TBD'))}
- **Goal:** {scene.get('goal', 'TBD')}
- **Conflict:** {scene.get('conflict', 'TBD')}
- **Turn:** {scene.get('turn', 'TBD')}
- **Location:** {scene.get('location', 'TBD')}

**Key Moments:**
"""
            for moment in scene.get('key_moments', ['*[Key moments TBD]*']):
                content += f"- {moment}\n"
            
            content += "\n"
    
    with open(filepath, 'w') as f:
        f.write(content)


def generate_foreshadowing_tracker(state: dict, filepath: str):
    """Generate foreshadowing tracker document."""
    content = f"""# Foreshadowing Tracker — {state.get('title', 'Untitled')}

"""
    
    for i, thread in enumerate(state.get('foreshadowing', []), 1):
        content += f"""## Thread {i}: {thread.get('name', 'Unnamed Thread')}

**Type:** {thread.get('type', 'TBD')}
**Final Payoff:** {thread.get('final_payoff', 'TBD')}

| Stage | Chapter | Scene | How |
|-------|---------|-------|-----|
"""
        for stage in thread.get('stages', []):
            content += f"| {stage.get('stage', '?')} | Ch {stage.get('chapter', '?')} | {stage.get('scene', '?')} | {stage.get('how', 'TBD')} |\n"
        
        content += "\n---\n\n"
    
    if not state.get('foreshadowing'):
        content += "*No foreshadowing threads defined yet.*\n"
    
    with open(filepath, 'w') as f:
        f.write(content)


def generate_character_threads(state: dict, filepath: str):
    """Generate character arc tracking document."""
    content = f"""# Character Threads — {state.get('title', 'Untitled')}

"""
    
    for char_name, char_data in state.get('character_threads', {}).items():
        content += f"""## {char_name}

### Arc Summary
- **Starts as:** {char_data.get('starts_as', 'TBD')}
- **Becomes:** {char_data.get('becomes', 'TBD')}
- **Key Change:** {char_data.get('key_change', 'TBD')}

### Chapter Presence

| Chapter | Role | Emotional State | Key Moment |
|---------|------|-----------------|------------|
"""
        for appearance in char_data.get('appearances', []):
            content += f"| {appearance.get('chapter', '?')} | {appearance.get('role', '?')} | {appearance.get('emotional_state', '?')} | {appearance.get('key_moment', '-')} |\n"
        
        content += "\n---\n\n"
    
    if not state.get('character_threads'):
        content += "*No character threads defined yet.*\n"
    
    with open(filepath, 'w') as f:
        f.write(content)


def generate_chapter_file(chapter: dict, state: dict, filepath: str):
    """Generate individual chapter outline file."""
    content = f"""# Chapter {chapter.get('number', '?')}: {chapter.get('title', 'Untitled')}

## Metadata
- **Beats:** {chapter.get('beats', 'TBD')}
- **Strand Focus:** {chapter.get('strand_focus', 'TBD')}
- **POV:** {chapter.get('pov', 'TBD')}
- **Location(s):** {chapter.get('locations', 'TBD')}
- **Word Target:** {chapter.get('word_target', 'TBD')}

## Purpose
{chapter.get('purpose', '*[Why this chapter exists]*')}

## Summary
{chapter.get('summary', '*[Chapter summary]*')}

---

## Opening
**Type:** {chapter.get('opening_type', 'TBD')}
**Hook:** {chapter.get('opening_hook', '*[What draws reader in]*')}

---

## Scenes

"""
    
    for i, scene in enumerate(chapter.get('scenes', []), 1):
        content += f"""### Scene {i}: {scene.get('title', 'Untitled')}

**Goal:** {scene.get('goal', 'TBD')}
**Conflict:** {scene.get('conflict', 'TBD')}
**Turn:** {scene.get('turn', 'TBD')}

**Characters:** {', '.join(scene.get('characters', ['TBD']))}
**Location:** {scene.get('location', 'TBD')}
**Tone:** {scene.get('tone', 'TBD')}

**Key Moments:**
"""
        for moment in scene.get('key_moments', ['*[TBD]*']):
            content += f"- {moment}\n"
        
        if scene.get('plants') or scene.get('payoffs'):
            content += "\n**Plants/Payoffs:**\n"
            for plant in scene.get('plants', []):
                content += f"- Plants: {plant}\n"
            for payoff in scene.get('payoffs', []):
                content += f"- Pays off: {payoff}\n"
        
        content += "\n---\n\n"
    
    content += f"""## Chapter Turn
**Type:** {chapter.get('turn_type', 'TBD')}
**The Turn:** {chapter.get('chapter_turn', '*[How chapter ends]*')}

## Connections
- **From Previous:** {chapter.get('from_previous', '*[Connection to Ch ' + str(chapter.get('number', 1) - 1) + ']*')}
- **To Next:** {chapter.get('to_next', '*[Connection to Ch ' + str(chapter.get('number', 1) + 1) + ']*')}

## Notes
{chapter.get('notes', '*[Additional notes]*')}
"""
    
    with open(filepath, 'w') as f:
        f.write(content)


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: compile_outline.py <path>")
        print("  path: Project directory")
        sys.exit(1)
    
    path = sys.argv[1]
    
    print(f"Compiling outline documents...")
    files = compile_outline(path)
    print()
    print(f"[OK] Generated {len(files)} documents in {os.path.join(path, 'outline')}")
