#!/usr/bin/env python3
"""Initialize a new Crucible outline project."""

import json
import os
import sys
from datetime import datetime


def get_template_dir() -> str:
    """Get the path to the plugin's templates directory."""
    # Script is at: skills/crucible-outliner/scripts/init_outline.py
    # Templates are at: templates/
    script_dir = os.path.dirname(os.path.abspath(__file__))
    plugin_root = os.path.dirname(os.path.dirname(os.path.dirname(script_dir)))
    return os.path.join(plugin_root, "templates")


def copy_template(template_path: str, dest_path: str, substitutions: dict = None) -> bool:
    """
    Copy a template file to destination with optional placeholder substitution.

    Args:
        template_path: Path to the template file
        dest_path: Destination path
        substitutions: Dict of {placeholder: value} for substitution

    Returns:
        True if copied successfully, False otherwise
    """
    if not os.path.exists(template_path):
        return False

    try:
        with open(template_path, "r", encoding="utf-8") as f:
            content = f.read()

        if substitutions:
            for placeholder, value in substitutions.items():
                content = content.replace(placeholder, value)

        with open(dest_path, "w", encoding="utf-8") as f:
            f.write(content)
        return True
    except (OSError, UnicodeDecodeError):
        return False


def init_outline(path: str, title: str, book_info: str = "Standalone") -> dict:
    """
    Initialize a new outline project directory.

    Args:
        path: Directory path for the project
        title: Book title
        book_info: "Standalone" or "Book X of Y"

    Returns:
        dict with project info
    """
    # Create directory structure
    os.makedirs(path, exist_ok=True)
    os.makedirs(os.path.join(path, "outline"), exist_ok=True)
    os.makedirs(os.path.join(path, "outline", "by-chapter"), exist_ok=True)

    # Create .crucible/state/ directory for state file
    state_dir = os.path.join(path, ".crucible", "state")
    os.makedirs(state_dir, exist_ok=True)

    # Initialize state
    state = {
        "title": title,
        "book_info": book_info,
        "created": datetime.now().isoformat(),
        "updated": datetime.now().isoformat(),
        "phase": "setup",
        "crucible_elements": {},
        "structure": {
            "chapter_count": None,
            "beat_mapping": [],
            "movements": {}
        },
        "chapters": [],
        "foreshadowing": [],
        "character_threads": {},
        "current_chapter": 0
    }

    # Save state to .crucible/state/outline-state.json
    state_path = os.path.join(state_dir, "outline-state.json")
    with open(state_path, "w") as f:
        json.dump(state, f, indent=2)

    # Copy outline CLAUDE.md template for context management
    template_dir = get_template_dir()
    templates_copied = []

    outline_template = os.path.join(template_dir, "project-structure", "outline", "CLAUDE.md")
    outline_dest = os.path.join(path, "outline", "CLAUDE.md")
    if copy_template(outline_template, outline_dest):
        templates_copied.append("outline/CLAUDE.md")

    print(f"[OK] Initialized outline project: {title}")
    print(f"   Location: {path}")
    print(f"   Book: {book_info}")
    print(f"   State file: {state_path}")
    if templates_copied:
        print(f"   Context templates: {', '.join(templates_copied)}")
    print()
    print("Directory structure created:")
    print(f"   {path}/")
    print("   +-- .crucible/")
    print("   |   +-- state/")
    print("   |       +-- outline-state.json")
    print("   +-- outline/")
    print("       +-- CLAUDE.md          (outline context)")
    print("       +-- by-chapter/")

    return state


if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("Usage: init_outline.py <path> <title> [book_info]")
        print("  path: Directory for the project")
        print("  title: Book title")
        print("  book_info: 'Standalone' or 'Book X of Y' (optional)")
        sys.exit(1)
    
    path = sys.argv[1]
    title = sys.argv[2]
    book_info = sys.argv[3] if len(sys.argv) > 3 else "Standalone"
    
    init_outline(path, title, book_info)
