#!/usr/bin/env python3
"""Save outline project state."""

import json
import os
import sys
from datetime import datetime


def find_state_file(path: str) -> str:
    """Find the outline state file, checking new location first then legacy."""
    # New location: .crucible/state/outline-state.json
    new_path = os.path.join(path, ".crucible", "state", "outline-state.json")
    if os.path.exists(new_path):
        return new_path

    # Legacy location: state.json at project root
    legacy_path = os.path.join(path, "state.json")
    if os.path.exists(legacy_path):
        return legacy_path

    return None


def get_state_path(path: str) -> str:
    """Get the state file path, preferring new location."""
    # Check if new structure exists
    new_dir = os.path.join(path, ".crucible", "state")
    if os.path.exists(new_dir):
        return os.path.join(new_dir, "outline-state.json")

    # Check if legacy file exists (use it for updates)
    legacy_path = os.path.join(path, "state.json")
    if os.path.exists(legacy_path):
        return legacy_path

    # Default to new location
    os.makedirs(new_dir, exist_ok=True)
    return os.path.join(new_dir, "outline-state.json")


def save_outline(path: str, updates: dict = None) -> dict:
    """
    Save or update outline project state.

    Args:
        path: Project directory path
        updates: Optional dict of state updates to merge

    Returns:
        Updated state dict
    """
    state_path = find_state_file(path)

    # Load existing state
    if state_path and os.path.exists(state_path):
        with open(state_path, "r") as f:
            state = json.load(f)
    else:
        raise FileNotFoundError(
            f"No state file found. Checked:\n"
            f"  - {os.path.join(path, '.crucible', 'state', 'outline-state.json')}\n"
            f"  - {os.path.join(path, 'state.json')}"
        )
    
    # Merge updates if provided
    if updates:
        state = deep_merge(state, updates)
    
    # Update timestamp
    state["updated"] = datetime.now().isoformat()
    
    # Save state (use same location it was loaded from)
    save_path = get_state_path(path) if not state_path else state_path
    with open(save_path, "w") as f:
        json.dump(state, f, indent=2)

    print(f"[OK] State saved: {save_path}")
    
    # Print progress summary
    if state.get("chapters"):
        completed = len([c for c in state["chapters"] if c.get("completed")])
        total = state["structure"].get("chapter_count", "?")
        print(f"   Progress: {completed}/{total} chapters outlined")
    
    return state


def deep_merge(base: dict, updates: dict) -> dict:
    """Deep merge two dictionaries."""
    result = base.copy()
    for key, value in updates.items():
        if key in result and isinstance(result[key], dict) and isinstance(value, dict):
            result[key] = deep_merge(result[key], value)
        else:
            result[key] = value
    return result


def add_chapter(path: str, chapter_data: dict) -> dict:
    """
    Add or update a chapter in the outline.

    Args:
        path: Project directory path
        chapter_data: Chapter outline data

    Returns:
        Updated state dict
    """
    state_path = find_state_file(path)
    if not state_path:
        raise FileNotFoundError(f"No state file found in {path}")

    with open(state_path, "r") as f:
        state = json.load(f)
    
    chapter_num = chapter_data.get("number", len(state["chapters"]) + 1)
    
    # Update or append chapter
    existing_idx = None
    for i, ch in enumerate(state["chapters"]):
        if ch.get("number") == chapter_num:
            existing_idx = i
            break
    
    if existing_idx is not None:
        state["chapters"][existing_idx] = chapter_data
        action = "Updated"
    else:
        state["chapters"].append(chapter_data)
        action = "Added"
    
    state["updated"] = datetime.now().isoformat()
    state["current_chapter"] = chapter_num
    
    with open(state_path, "w") as f:
        json.dump(state, f, indent=2)
    
    print(f"[OK] {action} Chapter {chapter_num}: {chapter_data.get('title', 'Untitled')}")
    
    return state


def add_foreshadowing(path: str, thread_data: dict) -> dict:
    """
    Add a foreshadowing thread.

    Args:
        path: Project directory path
        thread_data: Foreshadowing thread data

    Returns:
        Updated state dict
    """
    state_path = find_state_file(path)
    if not state_path:
        raise FileNotFoundError(f"No state file found in {path}")

    with open(state_path, "r") as f:
        state = json.load(f)
    
    state["foreshadowing"].append(thread_data)
    state["updated"] = datetime.now().isoformat()
    
    with open(state_path, "w") as f:
        json.dump(state, f, indent=2)
    
    print(f"[OK] Added foreshadowing thread: {thread_data.get('name', 'Unnamed')}")
    
    return state


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: save_outline.py <path> [updates_json]")
        print("  path: Project directory")
        print("  updates_json: Optional JSON string of state updates")
        sys.exit(1)
    
    path = sys.argv[1]
    updates = json.loads(sys.argv[2]) if len(sys.argv) > 2 else None
    
    save_outline(path, updates)
