# Context Management Guide

How to write a novel effectively within AI context constraints.

## The Core Problem

A typical novel is 80,000-150,000 words (~100,000-200,000 tokens). While current Claude models support 200,000 token context windows, long writing sessions accumulate:

- Conversation history
- Tool call outputs and results
- System prompts and skill instructions
- Previous drafts and revisions

**Practical available context** is typically 50-100K tokens after overhead.

**Solution:** Strategic loading via subtree CLAUDE.md files and disciplined state-saving.

## How Context Loading Works in Claude Code

> **Official Behavior** (from Claude Code documentation):
>
> 1. **Root CLAUDE.md** loads at session start (eagerly)
> 2. **Subtree CLAUDE.md** files (in subdirectories) load **lazily** — only when Claude **reads files** in that directory
>
> This means: Keep the root CLAUDE.md minimal. Phase-specific context lives in subtree CLAUDE.md files that load only when needed.

### Crucible Project Structure

```
project/
├── CLAUDE.md                    # Minimal - loads at startup (~50 lines)
├── story-bible.json             # NOT imported - queried on-demand
├── style-profile.json           # Loaded once per writing session
├── planning/
│   ├── CLAUDE.md                # Loads when reading files in planning/
│   └── [planning docs...]       # Referenced in CLAUDE.md guidance
├── outline/
│   ├── CLAUDE.md                # Loads when reading files in outline/
│   └── [outline docs...]        # Referenced in CLAUDE.md guidance
└── draft/
    ├── CLAUDE.md                # Loads when reading files in draft/
    └── chapters/
        └── [chapter files...]
```

### What This Achieves

| Phase | What Loads | Token Estimate |
|-------|------------|----------------|
| Session start | Root CLAUDE.md only | ~500 |
| Planning work | + planning/CLAUDE.md | ~1,000 |
| Outline work | + outline/CLAUDE.md | ~1,000 |
| Writing work | + draft/CLAUDE.md | ~1,000 |
| **Available for work** | **~85,000** | |

## The Context Budget

> **Note:** Token estimates below are **conservative planning heuristics**, not official specifications. System overhead (prompts, skills, tools) typically consumes 10-20K tokens depending on enabled features. We budget conservatively to ensure reliable operation during long writing sessions.

For each writing session, budget context approximately:

| Content | Token Estimate | Priority |
|---------|----------------|----------|
| System prompt + skill | ~12,000 | Required |
| Current scene outline | ~500 | Required |
| Style profile | ~1,000 | Required |
| Previous chapter summary | ~500 | Required |
| Character states (queried) | ~1,000 | Required |
| Relevant foreshadowing | ~500 | High |
| World details (if needed) | ~500 | Medium |
| **Working space for draft** | **~80,000** | Required |

*Note: System overhead (10-20K) varies by model and enabled features. The 80K working budget provides safety margin for this variability plus conversation accumulation.*

**Never load:**
- Full text of previous chapters (use summaries)
- Full planning documents (reference via subtree CLAUDE.md guidance)
- Multiple chapters at once
- Entire story bible JSON (query specific sections)

## What to Load Per Scene

### Minimum Required Context

```
1. CURRENT SCENE OUTLINE
   - Goal, conflict, turn
   - Key moments
   - Plants/payoffs for this scene

2. STYLE PROFILE
   - The extracted voice characteristics
   - Any author-confirmed adjustments

3. PREVIOUS CHAPTER SUMMARY (not full text)
   - 100-200 word summary
   - Final character states
   - Ending hook to continue from

4. ACTIVE CHARACTER STATES
   - Who is present this scene
   - Their current emotional state
   - What they know at this point
```

### Load On-Demand Only

```
5. SPECIFIC WORLD DETAILS (if scene requires)
   - Only the location being written
   - Only the relevant magic/power rules

6. SPECIFIC CHARACTER DETAILS (if introducing)
   - Full bio only for characters appearing for first time
   - Abbreviated after that

7. FORESHADOWING TRACKER (specific threads)
   - Only threads being planted or paid off this scene
```

## The Story Bible Strategy

The story bible is your memory system. It tracks everything written so you don't need to hold it in context.

### Querying the Story Bible

**DO NOT import the entire story-bible.json.** Query specific sections:

```python
# Get current character states
import json
with open('story-bible.json') as f:
    bible = json.load(f)
    print(bible.get('character_states', {}).get('CHARACTER_NAME'))

# Get chapter summary
print(bible.get('chapters', {}).get('CHAPTER_NUM', {}).get('summary'))

# Get unresolved foreshadowing
planted = bible.get('foreshadowing', {}).get('planted', [])
unresolved = [p for p in planted if not p.get('paid_off')]
```

### Story Bible Structure

Located at `story-bible.json` in project root. Key sections:

- `meta` - Title, targets, timestamps
- `progress` - Current chapter/scene/word count
- `chapters` - Per-chapter summaries and final states
- `character_states` - Current state of each character
- `established_facts` - Facts established in prose
- `foreshadowing.planted` / `foreshadowing.paid_off` - Thread tracking
- `timeline` - When each chapter occurs
- `invented_details` - Flagged inventions awaiting review
- `mercy_engine` - Mercy tracking for Crucible structure

### Updating the Story Bible

After EVERY scene:
1. Update character locations/states
2. Log any new established facts
3. Record any foreshadowing planted
4. Note any foreshadowing paid off
5. Update word count

After EVERY chapter:
1. Write chapter summary (100-200 words)
2. Record final character states
3. Update timeline
4. Flag any [INVENTED] details for review
5. Save full chapter text to file

## Resuming Work

When starting a new session:

### Quick Load Sequence

```
1. Root CLAUDE.md loads automatically (~500 tokens)
2. Read file in draft/ directory → draft/CLAUDE.md loads (~500 tokens)
3. Query style profile (~1,000 tokens)
4. Query current chapter outline (~500 tokens)
5. Query previous chapter summary from story bible (~200 tokens)
6. Ready to write → ~80,000 tokens available (conservative estimate)
```

*Actual availability depends on system overhead (~10-20K), conversation history, and model context limit (currently 200K).*

### What NOT to Reload

- Don't re-read completed chapters (use summaries)
- Don't reload full planning documents
- Don't reload the entire outline (only current chapter)
- Don't import story-bible.json (query specific sections)

## Emergency Recovery

If context is corrupted or session breaks:

### Recovery Protocol

```
1. Find last saved scene in draft/chapters/
2. Query story-bible.json for current state
3. Identify last complete scene
4. Load ONLY that chapter's outline
5. Re-load style profile
6. Resume from last checkpoint
```

### Prevention

- Save after EVERY scene (not just chapters)
- Story bible updates are atomic (one change at a time)
- Never trust context alone—always verify against saved state

## Multi-Chapter Context

Sometimes you need awareness of multiple chapters:

### When Writing a Payoff Scene

```
Load:
- Current scene outline
- The original plant (from foreshadowing tracker query)
- The plant scene SUMMARY (not full text)
- Style profile

Don't load:
- Full text of planting chapter
- Intervening chapters
```

### When Checking Continuity

```
Load:
- Current chapter outline
- Story bible character states (query)
- Story bible established facts (query)
- Last 3 chapter SUMMARIES (from story bible)

Don't load:
- Full text of any previous chapter
- Full planning documents
```

## Word Count Implications

Context limits affect word count targets:

| Scene Length | Can Write in One Pass |
|--------------|----------------------|
| 500-1,500 words | Yes, easily |
| 1,500-3,000 words | Yes, with care |
| 3,000-5,000 words | Consider splitting |
| 5,000+ words | Must split into multiple scenes |

**Strategy for long scenes:**
1. Write first half
2. Save checkpoint
3. Load summary of first half
4. Write second half
5. Merge in final compile

## Manual Context Compaction

During long writing sessions, conversation history accumulates and reduces available working space. Use the `/compact` command to reclaim context:

### When to Use /compact

- After completing a chapter (before starting the next)
- When Claude mentions context is getting full
- After lengthy review or revision discussions
- Before any operation requiring significant working space

### What /compact Does

The `/compact` command summarizes the conversation history into a condensed form, preserving key decisions and context while freeing up token space. This is especially useful during:

- Multi-chapter writing sessions
- Extended editing passes
- Long planning discussions

### Best Practice

```
1. Complete current scene/chapter
2. Save all state (story bible, draft file)
3. Run /compact
4. Resume writing with refreshed context
```

*Note: After compaction, Claude retains knowledge of what was discussed but loses verbatim conversation history. Always verify current state against saved files after compaction.*

## Key Principles

1. **Root CLAUDE.md is minimal** — Keep it short, phase-specific context goes in subdirectories
2. **Subtree CLAUDE.md for lazy loading** — Phase-specific context loads when reading files in that directory
3. **Story bible is memory** — Query it, don't import it
4. **Save after every scene** — Sessions break
5. **Load minimum needed** — Leave room for writing
6. **Verify against saved state** — Don't trust context alone
7. **Use /compact for long sessions** — Reclaim context when conversation grows
