#!/usr/bin/env python3
"""Initialize a new Crucible draft project."""

import json
import os
import sys
from datetime import datetime


def get_template_dir() -> str:
    """Get the path to the plugin's templates directory."""
    # Script is at: skills/crucible-writer/scripts/init_draft.py
    # Templates are at: templates/
    script_dir = os.path.dirname(os.path.abspath(__file__))
    plugin_root = os.path.dirname(os.path.dirname(os.path.dirname(script_dir)))
    return os.path.join(plugin_root, "templates")


def copy_template(template_path: str, dest_path: str, substitutions: dict = None) -> bool:
    """
    Copy a template file to destination with optional placeholder substitution.

    Args:
        template_path: Path to the template file
        dest_path: Destination path
        substitutions: Dict of {placeholder: value} for substitution

    Returns:
        True if copied successfully, False otherwise
    """
    if not os.path.exists(template_path):
        return False

    try:
        with open(template_path, "r", encoding="utf-8") as f:
            content = f.read()

        if substitutions:
            for placeholder, value in substitutions.items():
                content = content.replace(placeholder, value)

        with open(dest_path, "w", encoding="utf-8") as f:
            f.write(content)
        return True
    except (OSError, UnicodeDecodeError):
        return False


VOICE_SAMPLE_TEMPLATE = """# Voice Sample

This file helps the Crucible system maintain your unique narrative voice throughout the novel.

---

## Option 1: Prose Sample (Recommended)

Paste 2,000+ words of your best prose below. This can be:
- An excerpt from a previous published work
- A polished short story
- A scene you've written that captures your ideal voice

The more representative the sample, the better the system can match your style.

### Your Prose Sample

[Paste your prose sample here - aim for 2,000-5,000 words]

---

## Option 2: Style Preferences

If you don't have a prose sample, describe your preferences for each category below.

### Sentence Structure
- Average length: [short/medium/long/varied]
- Complexity: [simple/compound/complex/mixed]
- Rhythm: [staccato/flowing/varied]
- Fragment use: [never/rare/occasional/frequent]

### Vocabulary
- Register: [casual/literary/formal/archaic]
- Density: [spare/moderate/rich/ornate]
- Signature words/phrases: [list any recurring terms you favor]
- Words to avoid: [list words or phrases you dislike]

### Dialogue
- Attribution style: [said-only/varied-tags/minimal-tags/action-beats]
- Character voice distinction: [subtle/moderate/distinct]
- Subtext level: [on-the-nose/moderate/heavily-implied]
- Dialect/accent use: [none/light/moderate/heavy]

### Description
- Sensory focus: [visual/auditory/tactile/mixed]
- Metaphor density: [sparse/moderate/frequent/elaborate]
- Setting integration: [separate-blocks/woven-in/minimal]
- Detail level: [impressionistic/selective/comprehensive]

### Interiority (Character Thoughts)
- POV depth: [shallow/medium/deep/omniscient]
- Emotion approach: [shown/told/mixed]
- Reflection frequency: [rare/occasional/frequent]
- Internal voice style: [formal/casual/stream-of-consciousness]

### Pacing
- Scene transitions: [hard-cuts/bridging/time-stamps]
- White space use: [minimal/moderate/generous]
- Tension technique: [slow-build/peaks-valleys/sustained]
- Chapter length: [short/medium/long/varied]

### Signature Elements
List distinctive features of your writing style:
- [e.g., "Opening chapters with weather"]
- [e.g., "Using food as metaphor for relationships"]
- [e.g., "Ending scenes mid-action"]

### Authors to Emulate
List authors whose style you admire or want to echo:
- [Author name]: [specific element you like]
- [Author name]: [specific element you like]

### Things to Avoid
List stylistic choices you specifically want to prevent:
- [e.g., "Purple prose in action scenes"]
- [e.g., "Overuse of adverbs"]
- [e.g., "Info-dumping in dialogue"]

---

## Notes

- You can use both options: provide a sample AND add specific preferences
- Update this file anytime your style evolves
- The voice-checker agent references this during every writing session
"""


def init_draft(
    path: str,
    title: str,
    chapters: int = 25,
    target_words: int = 100000,
    book_number: int = 1,
    series_name: str = None,
    voice_sample: str = None
) -> dict:
    """
    Initialize a new draft project directory.

    Args:
        path: Directory path for the project
        title: Book title
        chapters: Expected chapter count
        target_words: Target word count
        book_number: Book number in series (1 for standalone)
        series_name: Series name (None for standalone)
        voice_sample: Path to existing voice sample file, or None to create template

    Returns:
        dict with project info
    """
    # Create directory structure
    os.makedirs(path, exist_ok=True)
    os.makedirs(os.path.join(path, ".crucible", "state"), exist_ok=True)
    os.makedirs(os.path.join(path, ".crucible", "backups"), exist_ok=True)
    os.makedirs(os.path.join(path, "draft", "chapters"), exist_ok=True)
    os.makedirs(os.path.join(path, "manuscript"), exist_ok=True)
    
    # Calculate words per chapter
    words_per_chapter = target_words // chapters
    
    # Initialize story bible
    story_bible = {
        "meta": {
            "title": title,
            "series": series_name,
            "book_number": book_number,
            "target_words": target_words,
            "target_chapters": chapters,
            "words_per_chapter": words_per_chapter,
            "created": datetime.now().isoformat(),
            "updated": datetime.now().isoformat()
        },
        "progress": {
            "current_chapter": 1,
            "current_scene": 1,
            "total_words": 0,
            "chapters_complete": 0,
            "status": "initialized"
        },
        "chapters": {},
        "character_states": {},
        "established_facts": [],
        "foreshadowing": {
            "planted": [],
            "paid_off": []
        },
        "timeline": {},
        "invented_details": [],
        "locations": {},
        "relationships": {},
        "mercy_engine": {
            "mercy_acts": [],
            "mercy_refused": [],
            "mercy_balance": 0
        }
    }
    
    # Initialize style profile (empty, to be filled)
    style_profile = {
        "meta": {
            "captured": False,
            "sample_source": None,
            "updated": datetime.now().isoformat()
        },
        "sentences": {
            "average_length": None,
            "variation": None,
            "structure_mix": None,
            "rhythm": None
        },
        "vocabulary": {
            "register": None,
            "density": None,
            "signature_words": [],
            "avoid_words": []
        },
        "dialogue": {
            "attribution_style": None,
            "character_distinction": None,
            "subtext_level": None
        },
        "description": {
            "sensory_focus": None,
            "metaphor_density": None,
            "setting_integration": None
        },
        "interiority": {
            "pov_depth": None,
            "emotion_approach": None,
            "reflection_frequency": None
        },
        "pacing": {
            "transition_style": None,
            "white_space": None,
            "tension_technique": None
        },
        "signature_elements": [],
        "genre_conventions": {
            "genre": None,
            "subgenre": None,
            "specific_conventions": []
        }
    }
    
    # Initialize draft state (in .crucible/state/)
    draft_state = {
        "title": title,
        "path": path,
        "created": datetime.now().isoformat(),
        "updated": datetime.now().isoformat(),
        "phase": "setup",
        "outline_loaded": False,
        "style_captured": False,
        "voice_sample_provided": voice_sample is not None,
        "current_chapter": 1,
        "current_scene": 1,
        "chapters_complete": 0,
        "word_count": 0,
        "target_words": target_words,
        "last_review_at_chapter": 0,
        "last_session": {
            "started": None,
            "ended": None,
            "words_written": 0,
            "scenes_completed": 0
        }
    }

    # Save files
    bible_path = os.path.join(path, "story-bible.json")
    style_path = os.path.join(path, "style-profile.json")
    state_path = os.path.join(path, ".crucible", "state", "draft-state.json")

    with open(bible_path, "w", encoding="utf-8") as f:
        json.dump(story_bible, f, indent=2)

    with open(style_path, "w", encoding="utf-8") as f:
        json.dump(style_profile, f, indent=2)

    # Create voice sample file
    voice_sample_path = os.path.join(path, "voice-sample.md")
    voice_sample_content = None
    if voice_sample:
        # Try to read content from provided file
        if os.path.exists(voice_sample):
            try:
                with open(voice_sample, "r", encoding="utf-8") as f:
                    voice_sample_content = f.read()
            except (OSError, UnicodeDecodeError) as e:
                print(f"[WARN] Could not read voice sample file: {e}")
                print("   Creating template instead.")
        else:
            print(f"[WARN] Voice sample file not found: {voice_sample}")
            print("   Creating template instead.")

    with open(voice_sample_path, "w", encoding="utf-8") as f:
        if voice_sample_content:
            f.write(f"# Voice Sample for {title}\n\n")
            f.write(f"*Imported from: {voice_sample}*\n\n")
            f.write("---\n\n")
            f.write(voice_sample_content)
        else:
            f.write(VOICE_SAMPLE_TEMPLATE)

    with open(state_path, "w", encoding="utf-8") as f:
        json.dump(draft_state, f, indent=2)

    # Copy draft CLAUDE.md template for context management
    template_dir = get_template_dir()
    templates_copied = []

    draft_template = os.path.join(template_dir, "project-structure", "draft", "CLAUDE.md")
    draft_dest = os.path.join(path, "draft", "CLAUDE.md")
    if copy_template(draft_template, draft_dest):
        templates_copied.append("draft/CLAUDE.md")

    # Check for existing outline directory (should exist from outliner phase)
    outline_dir = os.path.join(path, "outline")
    if not os.path.exists(outline_dir):
        os.makedirs(outline_dir, exist_ok=True)
        # Create placeholder for master outline
        outline_path = os.path.join(outline_dir, "master-outline.md")
        with open(outline_path, "w", encoding="utf-8") as f:
            f.write(f"# {title} — Master Outline\n\n")
            f.write("[Run /crucible-suite:crucible-outline first, or paste your outline here]\n")

    print(f"[OK] Initialized draft project: {title}")
    print(f"   Location: {path}")
    print(f"   Target: {target_words:,} words across {chapters} chapters")
    print(f"   Per chapter: ~{words_per_chapter:,} words")
    if templates_copied:
        print(f"   Context templates: {', '.join(templates_copied)}")
    print()
    print("Directory structure created:")
    print(f"   {path}/")
    print("   +-- .crucible/")
    print("   |   +-- state/")
    print("   |   |   +-- draft-state.json")
    print("   |   +-- backups/")
    print("   +-- story-bible.json")
    print("   +-- style-profile.json")
    print("   +-- voice-sample.md" + (" (imported)" if voice_sample_content else " (template)"))
    print("   +-- outline/")
    print("   |   +-- master-outline.md")
    print("   +-- draft/")
    print("   |   +-- CLAUDE.md          (writing context)")
    print("   |   +-- chapters/")
    print("   +-- manuscript/")
    print()
    print("Next steps:")
    print("   1. Run /crucible-outline or load chapter outline")
    print("   2. Capture your writing style")
    print("   3. Begin drafting chapter 1")

    return {
        "story_bible": story_bible,
        "style_profile": style_profile,
        "draft_state": draft_state
    }


# Note: load_draft() functionality is in separate load_draft.py script


if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("Usage: init_draft.py <path> <title> [options]")
        print()
        print("Arguments:")
        print("  path                  Directory for the project")
        print("  title                 Book title")
        print()
        print("Options:")
        print("  --chapters N          Expected chapter count (default: 25)")
        print("  --target-words N      Target word count (default: 100000)")
        print("  --voice-sample FILE   Path to existing prose sample file")
        print()
        print("Examples:")
        print("  init_draft.py ./my-novel \"The Dark Tower\" --chapters 30")
        print("  init_draft.py ./my-novel \"The Dark Tower\" --voice-sample ./sample.md")
        sys.exit(1)

    path = sys.argv[1]
    title = sys.argv[2]

    # Parse optional arguments
    chapters = 25
    target_words = 100000
    voice_sample = None

    i = 3
    while i < len(sys.argv):
        if sys.argv[i] == "--chapters" and i + 1 < len(sys.argv):
            chapters = int(sys.argv[i + 1])
            i += 2
        elif sys.argv[i] == "--target-words" and i + 1 < len(sys.argv):
            target_words = int(sys.argv[i + 1])
            i += 2
        elif sys.argv[i] == "--voice-sample" and i + 1 < len(sys.argv):
            voice_sample = sys.argv[i + 1]
            i += 2
        else:
            i += 1

    init_draft(path, title, chapters, target_words, voice_sample=voice_sample)
