#!/usr/bin/env python3
"""Save draft progress and chapter content."""

import json
import os
import sys
from datetime import datetime

# Import shared state sync function to avoid duplication
from update_story_bible import sync_draft_state, load_bible


def save_scene(
    path: str, 
    chapter: int, 
    scene: int, 
    content: str,
    scene_summary: str = None
) -> dict:
    """
    Save a scene draft.
    
    Args:
        path: Project directory
        chapter: Chapter number
        scene: Scene number
        content: Scene prose content
        scene_summary: Brief summary for story bible
    
    Returns:
        Updated progress info
    """
    # Load project state
    bible_path = os.path.join(path, "story-bible.json")

    # Find draft state file (new location first, then legacy)
    state_path = os.path.join(path, ".crucible", "state", "draft-state.json")
    if not os.path.exists(state_path):
        # Try legacy location
        state_path = os.path.join(path, "project-state.json")

    # Load story bible using load_bible() for automatic schema validation
    # This ensures backward compatibility with older projects missing new sections
    try:
        story_bible = load_bible(path)
    except FileNotFoundError:
        raise FileNotFoundError(f"Story bible not found at {bible_path}. Run init_draft.py first.")
    except ValueError as e:
        raise ValueError(f"Invalid JSON in story-bible.json: {e}")

    # Load draft state with error handling
    if not os.path.exists(state_path):
        raise FileNotFoundError(f"Draft state not found. Checked:\n"
                                f"  - {os.path.join(path, '.crucible', 'state', 'draft-state.json')}\n"
                                f"  - {os.path.join(path, 'project-state.json')}\n"
                                f"Run init_draft.py first.")

    try:
        with open(state_path, "r", encoding="utf-8") as f:
            draft_state = json.load(f)
    except json.JSONDecodeError as e:
        raise ValueError(f"Invalid JSON in draft state: {e}")
    
    # Create chapter file if needed
    chapter_dir = os.path.join(path, "draft", "chapters")
    chapter_file = os.path.join(chapter_dir, f"ch{chapter:02d}.md")
    
    # Read existing chapter content or create new
    if os.path.exists(chapter_file):
        with open(chapter_file, "r", encoding="utf-8") as f:
            chapter_content = f.read()
    else:
        chapter_content = f"# Chapter {chapter}\n\n"
    
    # Add scene marker and content
    scene_marker = f"\n## Scene {scene}\n\n"
    
    # If scene already exists, replace it; otherwise append
    if scene_marker in chapter_content:
        # Find and replace the scene
        start = chapter_content.find(scene_marker)
        next_scene = chapter_content.find(f"\n## Scene {scene + 1}\n", start)
        if next_scene == -1:
            # Last scene in chapter
            chapter_content = chapter_content[:start] + scene_marker + content + "\n"
        else:
            chapter_content = chapter_content[:start] + scene_marker + content + "\n" + chapter_content[next_scene:]
    else:
        # Append new scene
        chapter_content += scene_marker + content + "\n"
    
    # Save chapter file
    with open(chapter_file, "w", encoding="utf-8") as f:
        f.write(chapter_content)
    
    # Count words in scene
    word_count = len(content.split())
    
    # Update story bible
    chapter_key = str(chapter)
    if chapter_key not in story_bible["chapters"]:
        story_bible["chapters"][chapter_key] = {
            "title": f"Chapter {chapter}",
            "word_count": 0,
            "scenes": {},
            "summary": "",
            "final_state": {},
            "completed": False,
            "completed_at": None
        }
    
    story_bible["chapters"][chapter_key]["scenes"][str(scene)] = {
        "word_count": word_count,
        "summary": scene_summary or "",
        "saved": datetime.now().isoformat()
    }

    # Ensure completed_at field exists for schema consistency
    if "completed_at" not in story_bible["chapters"][chapter_key]:
        story_bible["chapters"][chapter_key]["completed_at"] = None

    # Recalculate chapter word count
    total_chapter_words = sum(
        s["word_count"] for s in story_bible["chapters"][chapter_key]["scenes"].values()
    )
    story_bible["chapters"][chapter_key]["word_count"] = total_chapter_words
    
    # Recalculate total words (filter out template entries like _example)
    total_words = sum(
        ch["word_count"] for k, ch in story_bible["chapters"].items()
        if not k.startswith("_")
    )
    story_bible["progress"]["total_words"] = total_words
    story_bible["progress"]["current_chapter"] = chapter
    story_bible["progress"]["current_scene"] = scene
    story_bible["meta"]["updated"] = datetime.now().isoformat()
    
    # Save story bible first
    with open(bible_path, "w", encoding="utf-8") as f:
        json.dump(story_bible, f, indent=2)

    # Use shared sync function to update draft state
    # This ensures consistency with complete_chapter() sync logic
    sync_draft_state(path, story_bible)
    
    print(f"[OK] Saved Chapter {chapter}, Scene {scene}")
    print(f"   Words in scene: {word_count:,}")
    print(f"   Chapter total: {total_chapter_words:,}")
    print(f"   Book total: {total_words:,}")
    
    return {
        "scene_words": word_count,
        "chapter_words": total_chapter_words,
        "total_words": total_words
    }


def complete_chapter(
    path: str,
    chapter: int,
    chapter_summary: str,
    final_character_states: dict = None
) -> dict:
    """
    Mark a chapter as complete and save summary.

    Also updates draft-state.json to keep state synchronized and
    checks if a bi-chapter review is needed.

    Args:
        path: Project directory
        chapter: Chapter number
        chapter_summary: 100-200 word summary
        final_character_states: Character states at chapter end

    Returns:
        Updated progress info including review_needed flag
    """
    bible_path = os.path.join(path, "story-bible.json")

    # Load story bible using load_bible() for automatic schema validation
    # This ensures backward compatibility with older projects missing new sections
    try:
        story_bible = load_bible(path)
    except FileNotFoundError:
        raise FileNotFoundError(f"Story bible not found at {bible_path}")
    except ValueError as e:
        raise ValueError(f"Invalid JSON in story-bible.json: {e}")

    chapter_key = str(chapter)

    if chapter_key not in story_bible["chapters"]:
        raise ValueError(f"Chapter {chapter} not found")

    story_bible["chapters"][chapter_key]["summary"] = chapter_summary
    story_bible["chapters"][chapter_key]["completed"] = True
    story_bible["chapters"][chapter_key]["completed_at"] = datetime.now().isoformat()

    if final_character_states:
        story_bible["chapters"][chapter_key]["final_state"] = final_character_states
        # Also update current character states WITH HISTORY TRACKING
        for char, state in final_character_states.items():
            if char not in story_bible["character_states"]:
                story_bible["character_states"][char] = {
                    "location": None,
                    "emotional_state": None,
                    "knows": [],
                    "inventory": [],
                    "relationships": {},
                    "history": []
                }

            # Record current state to history before updating
            current_state = dict(story_bible["character_states"][char])
            current_state.pop("history", None)  # Don't include history in history
            if "history" not in story_bible["character_states"][char]:
                story_bible["character_states"][char]["history"] = []
            story_bible["character_states"][char]["history"].append({
                "chapter": chapter,
                "state": current_state
            })

            # Now apply the updates
            story_bible["character_states"][char].update(state)

    # Update progress in story bible
    # Filter out template entries like _example when counting completed chapters
    completed = sum(1 for k, ch in story_bible["chapters"].items() if not k.startswith("_") and ch.get("completed", False))
    story_bible["progress"]["chapters_complete"] = completed
    story_bible["progress"]["current_chapter"] = chapter + 1
    story_bible["progress"]["current_scene"] = 1
    story_bible["meta"]["updated"] = datetime.now().isoformat()

    with open(bible_path, "w", encoding="utf-8") as f:
        json.dump(story_bible, f, indent=2)

    # === SYNC WITH DRAFT-STATE.JSON ===
    # Use the shared sync_draft_state() function from update_story_bible.py
    # This is the SINGLE SOURCE OF TRUTH for state synchronization
    sync_result = sync_draft_state(path, story_bible, check_review=True)

    review_needed = sync_result["review_needed"]
    review_chapters = sync_result["review_chapters"]

    target = story_bible["meta"]["target_chapters"]

    print(f"[OK] Chapter {chapter} marked complete")
    print(f"   Progress: {completed}/{target} chapters ({completed/target*100:.1f}%)")
    print(f"   Total words: {story_bible['progress']['total_words']:,}")

    if review_needed:
        print(f"")
        print(f"[WARN] BI-CHAPTER REVIEW NEEDED")
        print(f"   Review chapters {review_chapters[0]}-{review_chapters[1]} before continuing")
        print(f"   Run: /crucible-review {review_chapters[0]}-{review_chapters[1]}")

    return {
        "chapters_complete": completed,
        "total_words": story_bible["progress"]["total_words"],
        "review_needed": review_needed,
        "review_chapters": review_chapters
    }


def save_session_end(path: str, words_written: int, scenes_completed: int) -> None:
    """
    Save end-of-session information.

    Args:
        path: Project directory
        words_written: Words written this session
        scenes_completed: Scenes completed this session
    """
    # Find draft state file (new location first, then legacy)
    state_path = os.path.join(path, ".crucible", "state", "draft-state.json")
    if not os.path.exists(state_path):
        state_path = os.path.join(path, "project-state.json")

    with open(state_path, "r", encoding="utf-8") as f:
        draft_state = json.load(f)

    draft_state["last_session"]["ended"] = datetime.now().isoformat()
    draft_state["last_session"]["words_written"] = words_written
    draft_state["last_session"]["scenes_completed"] = scenes_completed
    draft_state["updated"] = datetime.now().isoformat()

    with open(state_path, "w", encoding="utf-8") as f:
        json.dump(draft_state, f, indent=2)

    print(f"[OK] Session saved")
    print(f"   Words this session: {words_written:,}")
    print(f"   Scenes completed: {scenes_completed}")


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage:")
        print("  save_draft.py <path> --scene <chapter> <scene> <content_file>")
        print("  save_draft.py <path> --complete-chapter <chapter> <summary_file>")
        print("  save_draft.py <path> --end-session <words> <scenes>")
        sys.exit(1)
    
    path = sys.argv[1]
    
    if len(sys.argv) > 2:
        if sys.argv[2] == "--scene" and len(sys.argv) >= 6:
            chapter = int(sys.argv[3])
            scene = int(sys.argv[4])
            with open(sys.argv[5], "r") as f:
                content = f.read()
            save_scene(path, chapter, scene, content)
        elif sys.argv[2] == "--complete-chapter" and len(sys.argv) >= 5:
            chapter = int(sys.argv[3])
            with open(sys.argv[4], "r") as f:
                summary = f.read()
            complete_chapter(path, chapter, summary)
        elif sys.argv[2] == "--end-session" and len(sys.argv) >= 5:
            words = int(sys.argv[3])
            scenes = int(sys.argv[4])
            save_session_end(path, words, scenes)
