#!/usr/bin/env python3
"""Update the story bible with new facts, foreshadowing, and character states."""

import json
import os
import sys
from datetime import datetime


# Valid categories for invented details (used for validation warnings)
INVENTION_CATEGORIES = [
    "character",
    "location",
    "item",
    "power",
    "creature",
    "culture",
    "history",
    "general"
]


# Complete schema for story-bible.json files.
# Used by ensure_schema() to validate and migrate story bibles on load.
STORY_BIBLE_SCHEMA = {
    "meta": {
        "title": "",
        "series": "",
        "book_number": 1,
        "target_words": 100000,
        "target_chapters": 25,
        "words_per_chapter": 4000,
        "created": "",
        "updated": ""
    },
    "progress": {
        "current_chapter": 1,
        "current_scene": 1,
        "total_words": 0,
        "chapters_complete": 0,
        "status": "initialized"
    },
    "chapters": {},
    "character_states": {},
    "established_facts": [],
    "foreshadowing": {
        "planted": [],
        "paid_off": []
    },
    "timeline": {},
    "invented_details": [],
    "locations": {},
    "relationships": {},
    "mercy_engine": {
        "mercy_acts": [],
        "mercy_refused": [],
        "mercy_balance": 0
    }
}


def ensure_schema(bible: dict) -> dict:
    """
    Validate and upgrade a story bible to include all required fields.

    This function ensures backward compatibility by adding any missing
    top-level sections and structural nested keys from STORY_BIBLE_SCHEMA.
    User-data sections (chapters, character_states, etc.) only get the
    top-level key added if missing - their contents are not modified.

    Args:
        bible: A loaded story bible dictionary

    Returns:
        The validated/upgraded bible dict with all required fields present
    """
    def merge_defaults(target: dict, defaults: dict, deep_keys: set) -> dict:
        """Recursively merge defaults into target for specified keys."""
        for key, default_value in defaults.items():
            if key not in target:
                # Key missing entirely - add default
                if isinstance(default_value, dict):
                    target[key] = dict(default_value)
                elif isinstance(default_value, list):
                    target[key] = list(default_value)
                else:
                    target[key] = default_value
            elif key in deep_keys and isinstance(default_value, dict):
                # Structural section - ensure nested keys exist
                if isinstance(target[key], dict):
                    for nested_key, nested_default in default_value.items():
                        if nested_key not in target[key]:
                            if isinstance(nested_default, dict):
                                target[key][nested_key] = dict(nested_default)
                            elif isinstance(nested_default, list):
                                target[key][nested_key] = list(nested_default)
                            else:
                                target[key][nested_key] = nested_default
        return target

    # Structural sections where we also ensure nested keys exist
    structural_sections = {"meta", "progress", "foreshadowing", "mercy_engine"}

    return merge_defaults(bible, STORY_BIBLE_SCHEMA, structural_sections)


def _filter_examples(data: dict) -> dict:
    """
    Filter out template example entries from story bible data.

    Removes keys starting with '_' (like '_example', '_example_character')
    and items containing '_comment' field.

    Args:
        data: Dict from story bible (e.g., chapters, character_states)

    Returns:
        Filtered dict without example/template entries
    """
    if not isinstance(data, dict):
        return data

    return {
        k: v for k, v in data.items()
        if not k.startswith("_") and not (isinstance(v, dict) and "_comment" in v)
    }


def _normalize_knows(knows_list: list) -> list:
    """
    Normalize knows list to structured format for backward compatibility.

    Converts old flat string format to new structured format:
    - Old: ["fact1", "fact2"]
    - New: [{"fact": "fact1", "learned_in": null}, {"fact": "fact2", "learned_in": null}]

    If already in new format, returns unchanged.

    Args:
        knows_list: List of knowledge items (strings or dicts)

    Returns:
        List of structured knowledge dicts
    """
    if not knows_list:
        return []

    normalized = []
    for item in knows_list:
        if isinstance(item, str):
            # Old format: convert to new
            normalized.append({
                "fact": item,
                "learned_in": None
            })
        elif isinstance(item, dict) and "fact" in item:
            # Already new format
            normalized.append(item)
        else:
            # Unknown format, wrap as fact
            normalized.append({
                "fact": str(item),
                "learned_in": None
            })

    return normalized


def load_bible(path: str) -> dict:
    """
    Load the story bible.

    Args:
        path: Project directory containing story-bible.json

    Returns:
        dict: The loaded story bible

    Raises:
        FileNotFoundError: If story-bible.json doesn't exist
        ValueError: If story-bible.json contains invalid JSON
    """
    bible_path = os.path.join(path, "story-bible.json")

    if not os.path.exists(bible_path):
        raise FileNotFoundError(
            f"Story bible not found at {bible_path}. "
            f"Run init_draft.py first to initialize the project."
        )

    try:
        with open(bible_path, "r", encoding="utf-8") as f:
            bible = json.load(f)
        return ensure_schema(bible)
    except json.JSONDecodeError as e:
        raise ValueError(f"Invalid JSON in story-bible.json: {e}")


def save_bible(path: str, bible: dict) -> None:
    """Save the story bible."""
    bible["meta"]["updated"] = datetime.now().isoformat()
    bible_path = os.path.join(path, "story-bible.json")
    with open(bible_path, "w", encoding="utf-8") as f:
        json.dump(bible, f, indent=2)


def sync_draft_state(path: str, bible: dict, check_review: bool = False) -> dict:
    """
    Sync draft-state.json with story-bible.json to prevent desynchronization.

    This is the SINGLE SOURCE OF TRUTH for state synchronization between
    story-bible.json and draft-state.json. All code that needs to sync these
    files should call this function rather than duplicating the logic.

    This ensures bi-chapter review tracking stays accurate when story bible
    is updated directly via update_story_bible.py.

    Args:
        path: Project directory path
        bible: The story bible dict (already loaded)
        check_review: If True, check if bi-chapter review is needed

    Returns:
        dict with sync results:
        {
            "synced": bool,           # Whether sync was successful
            "chapters_complete": int, # Number of completed chapters
            "review_needed": bool,    # True if bi-chapter review is due
            "review_chapters": tuple, # (start, end) chapters to review, or None
            "state_path": str         # Path to the draft state file
        }
    """
    result = {
        "synced": False,
        "chapters_complete": 0,
        "review_needed": False,
        "review_chapters": None,
        "state_path": None
    }

    # Find draft state file (new location first, then legacy)
    state_path = os.path.join(path, ".crucible", "state", "draft-state.json")
    if not os.path.exists(state_path):
        legacy_path = os.path.join(path, "project-state.json")
        if os.path.exists(legacy_path):
            state_path = legacy_path
        else:
            # No draft state file exists yet - nothing to sync
            return result

    result["state_path"] = state_path

    try:
        with open(state_path, "r", encoding="utf-8") as f:
            draft_state = json.load(f)
    except (json.JSONDecodeError, OSError):
        return result  # Can't sync if we can't read the state file

    # Sync progress fields from story bible
    progress = bible.get("progress", {})
    draft_state["current_chapter"] = progress.get("current_chapter", 1)
    draft_state["current_scene"] = progress.get("current_scene", 1)
    draft_state["word_count"] = progress.get("total_words", 0)

    # Sync chapters_complete by counting completed chapters
    chapters = bible.get("chapters", {})
    chapters_complete = sum(
        1 for k, ch in chapters.items()
        if not k.startswith("_") and ch.get("completed", False)
    )
    draft_state["chapters_complete"] = chapters_complete
    result["chapters_complete"] = chapters_complete

    # Check if bi-chapter review is needed
    # CANONICAL LOGIC: Review is needed when 2+ chapters completed since last review.
    # This logic is mirrored in scripts/backup_on_change.py:check_review_status()
    # If you change this logic, update that function to match.
    if check_review:
        last_review = draft_state.get("last_review_at_chapter", 0)
        chapters_since_review = chapters_complete - last_review

        if chapters_since_review >= 2:
            result["review_needed"] = True
            result["review_chapters"] = (last_review + 1, chapters_complete)
            draft_state["review_pending"] = True

    draft_state["updated"] = datetime.now().isoformat()

    try:
        with open(state_path, "w", encoding="utf-8") as f:
            json.dump(draft_state, f, indent=2)
        result["synced"] = True
    except OSError:
        pass  # Best effort sync - don't fail if we can't write

    return result


# Keep the old name as an alias for backwards compatibility
def _sync_draft_state(path: str, bible: dict) -> None:
    """Deprecated: Use sync_draft_state() instead."""
    sync_draft_state(path, bible, check_review=False)


def mark_review_complete(path: str, chapters_reviewed: int = None) -> dict:
    """
    Mark bi-chapter review as complete.

    Args:
        path: Project directory
        chapters_reviewed: Number of chapters reviewed (if None, uses chapters_complete)

    Returns:
        dict with updated state info
    """
    result = {"success": False, "last_review_at_chapter": 0}

    # Find draft state file (new location first, then legacy)
    state_path = os.path.join(path, ".crucible", "state", "draft-state.json")
    if not os.path.exists(state_path):
        legacy_path = os.path.join(path, "project-state.json")
        if os.path.exists(legacy_path):
            state_path = legacy_path
        else:
            print("Error: No draft state file found")
            return result

    try:
        with open(state_path, "r", encoding="utf-8") as f:
            draft_state = json.load(f)
    except (json.JSONDecodeError, OSError) as e:
        print(f"Error reading draft state: {e}")
        return result

    # Determine chapters_reviewed value
    if chapters_reviewed is None:
        chapters_reviewed = draft_state.get("chapters_complete", 0)

    # Update review tracking fields
    draft_state["review_pending"] = False
    draft_state["last_review_at_chapter"] = chapters_reviewed
    draft_state["updated"] = datetime.now().isoformat()

    try:
        with open(state_path, "w", encoding="utf-8") as f:
            json.dump(draft_state, f, indent=2)
        result["success"] = True
        result["last_review_at_chapter"] = chapters_reviewed
        print(f"Review complete. Last review at chapter: {chapters_reviewed}")
    except OSError as e:
        print(f"Error saving draft state: {e}")

    return result


def add_established_fact(path: str, fact: str, chapter: int, scene: int = None) -> None:
    """
    Record a fact established in the narrative.
    
    Args:
        path: Project directory
        fact: The fact that was established
        chapter: Chapter where established
        scene: Scene where established (optional)
    """
    bible = load_bible(path)
    
    location = f"ch{chapter}" + (f".{scene}" if scene else "")
    
    bible["established_facts"].append({
        "fact": fact,
        "established_in": location,
        "added": datetime.now().isoformat()
    })
    
    save_bible(path, bible)
    print(f"[OK] Added established fact: {fact[:50]}...")


def remove_established_fact(
    path: str,
    index: int = None,
    match_text: str = None
) -> bool:
    """
    Remove an established fact by index or matching text.

    Args:
        path: Project directory
        index: Index of fact to remove (0-based, excludes template entries)
        match_text: Text substring to match against fact text (case-insensitive)

    Returns:
        True if a fact was removed, False otherwise

    Note:
        Either index or match_text must be provided, not both.
        If match_text matches multiple facts, only the first is removed.
    """
    if index is None and match_text is None:
        print("[WARN] Error: Must provide either index or match_text")
        return False

    if index is not None and match_text is not None:
        print("[WARN] Error: Provide either index or match_text, not both")
        return False

    bible = load_bible(path)

    # Filter out template/comment entries for accurate indexing
    facts = bible.get("established_facts", [])
    real_facts = [(i, f) for i, f in enumerate(facts) if "_comment" not in f]

    removed_fact = None
    original_index = None

    if index is not None:
        # Remove by index (index refers to real facts, not including comments)
        if index < 0 or index >= len(real_facts):
            print(f"[WARN] Invalid index {index}. Valid range: 0-{len(real_facts) - 1}")
            return False
        original_index, removed_fact = real_facts[index]
    else:
        # Remove by matching text (case-insensitive)
        for orig_idx, fact in real_facts:
            if match_text.lower() in fact.get("fact", "").lower():
                original_index = orig_idx
                removed_fact = fact
                break

        if removed_fact is None:
            print(f"[WARN] No fact found matching: {match_text[:50]}...")
            return False

    # Remove from the list using original index
    del bible["established_facts"][original_index]

    save_bible(path, bible)
    print(f"[OK] Removed established fact: {removed_fact.get('fact', '')[:50]}...")
    return True


def add_foreshadowing_plant(
    path: str, 
    thread: str, 
    chapter: int, 
    scene: int,
    expected_payoff: str = None
) -> None:
    """
    Record a foreshadowing plant.
    
    Args:
        path: Project directory
        thread: Description of what was planted
        chapter: Chapter where planted
        scene: Scene where planted
        expected_payoff: Where/when this should pay off
    """
    bible = load_bible(path)
    
    bible["foreshadowing"]["planted"].append({
        "thread": thread,
        "planted_in": f"ch{chapter}.{scene}",
        "expected_payoff": expected_payoff,
        "paid_off": False,
        "added": datetime.now().isoformat()
    })
    
    save_bible(path, bible)
    print(f"[OK] Recorded foreshadowing plant: {thread[:50]}...")


def remove_foreshadowing_plant(path: str, match_thread: str) -> bool:
    """
    Remove an unresolved foreshadowing plant by matching thread text.

    Args:
        path: Project directory
        match_thread: Text substring to match against thread description (case-insensitive)

    Returns:
        True if a plant was removed, False otherwise

    Note:
        Only removes plants that have NOT been paid off (paid_off=False).
        If match_thread matches multiple plants, only the first is removed.
    """
    bible = load_bible(path)

    planted = bible.get("foreshadowing", {}).get("planted", [])

    # Find matching unresolved plant (excluding comments and paid-off plants)
    removed_plant = None
    removed_index = None

    for i, plant in enumerate(planted):
        # Skip template/comment entries
        if "_comment" in plant:
            continue
        # Skip already paid off
        if plant.get("paid_off", False):
            continue
        # Check for match (case-insensitive)
        if match_thread.lower() in plant.get("thread", "").lower():
            removed_plant = plant
            removed_index = i
            break

    if removed_plant is None:
        print(f"[WARN] No unresolved plant found matching: {match_thread[:50]}...")
        return False

    # Remove from the list
    del bible["foreshadowing"]["planted"][removed_index]

    save_bible(path, bible)
    print(f"[OK] Removed foreshadowing plant: {removed_plant.get('thread', '')[:50]}...")
    return True


def record_payoff(path: str, thread: str, chapter: int, scene: int) -> None:
    """
    Record that a foreshadowing thread was paid off.

    Args:
        path: Project directory
        thread: Description matching the original plant (exact or substring match)
        chapter: Chapter where paid off
        scene: Scene where paid off
    """
    bible = load_bible(path)

    # Find matching plants (excluding template _comment entries)
    matches = []
    for plant in bible["foreshadowing"]["planted"]:
        # Skip template/comment entries
        if "_comment" in plant:
            continue
        # Skip already paid off
        if plant.get("paid_off"):
            continue
        # Check for match (case-insensitive)
        if thread.lower() in plant["thread"].lower():
            matches.append(plant)

    if not matches:
        print(f"[WARN] Could not find matching plant for: {thread[:50]}...")
        return

    if len(matches) > 1:
        # Multiple matches - warn and show options
        print(f"[WARN] Multiple plants match '{thread[:30]}...':")
        for i, m in enumerate(matches):
            print(f"   [{i}] {m['thread'][:60]} (planted: {m.get('planted_in', '?')})")
        print(f"   Using first match. For precise matching, use the exact thread text.")

    # Use first match (or only match)
    plant = matches[0]
    plant["paid_off"] = True
    plant["paid_in"] = f"ch{chapter}.{scene}"

    # Also record in paid_off list
    bible["foreshadowing"]["paid_off"].append({
        "thread": plant["thread"],
        "planted_in": plant["planted_in"],
        "paid_in": f"ch{chapter}.{scene}",
        "recorded": datetime.now().isoformat()
    })

    save_bible(path, bible)
    print(f"[OK] Recorded payoff for: {plant['thread'][:50]}...")


def update_character_state(
    path: str, 
    character: str, 
    updates: dict,
    chapter: int = None
) -> None:
    """
    Update a character's current state.
    
    Args:
        path: Project directory
        character: Character name
        updates: Dict of state updates (location, emotional_state, knows, etc.)
        chapter: Chapter of update (for tracking)
    """
    bible = load_bible(path)
    
    if character not in bible["character_states"]:
        bible["character_states"][character] = {
            "location": None,
            "emotional_state": None,
            "knows": [],
            "inventory": [],
            "relationships": {},
            "history": []
        }
    
    # Record current state to history if changing
    if chapter:
        # Create a copy of current state WITHOUT history to prevent exponential growth
        current_state = dict(bible["character_states"][character])
        current_state.pop("history", None)  # Don't include history in history entry

        # Normalize 'knows' to structured format for consistent history
        if "knows" in current_state:
            current_state["knows"] = _normalize_knows(current_state["knows"])

        history_entry = {
            "chapter": chapter,
            "state": current_state
        }
        if "history" not in bible["character_states"][character]:
            bible["character_states"][character]["history"] = []
        bible["character_states"][character]["history"].append(history_entry)
    
    # Apply updates
    for key, value in updates.items():
        if key == "knows":
            # Normalize existing knows to structured format (backward compat)
            if "knows" not in bible["character_states"][character]:
                bible["character_states"][character]["knows"] = []
            else:
                bible["character_states"][character]["knows"] = _normalize_knows(
                    bible["character_states"][character]["knows"]
                )

            # Handle input: string, dict with fact, or list
            if isinstance(value, str):
                bible["character_states"][character]["knows"].append({
                    "fact": value,
                    "learned_in": f"ch{chapter}" if chapter else None
                })
            elif isinstance(value, dict) and "fact" in value:
                if "learned_in" not in value and chapter:
                    value["learned_in"] = f"ch{chapter}"
                bible["character_states"][character]["knows"].append(value)
            elif isinstance(value, list):
                for item in value:
                    if isinstance(item, str):
                        bible["character_states"][character]["knows"].append({
                            "fact": item,
                            "learned_in": f"ch{chapter}" if chapter else None
                        })
                    elif isinstance(item, dict):
                        bible["character_states"][character]["knows"].append(item)
        elif key == "inventory":
            # Handle inventory updates
            if "inventory" not in bible["character_states"][character]:
                bible["character_states"][character]["inventory"] = []

            if isinstance(value, str):
                bible["character_states"][character]["inventory"].append({
                    "item": value,
                    "acquired_in": f"ch{chapter}" if chapter else None,
                    "status": "possessed"
                })
            elif isinstance(value, dict) and "item" in value:
                if "acquired_in" not in value and chapter:
                    value["acquired_in"] = f"ch{chapter}"
                if "status" not in value:
                    value["status"] = "possessed"
                bible["character_states"][character]["inventory"].append(value)
            elif isinstance(value, list):
                for item in value:
                    if isinstance(item, str):
                        bible["character_states"][character]["inventory"].append({
                            "item": item,
                            "acquired_in": f"ch{chapter}" if chapter else None,
                            "status": "possessed"
                        })
                    elif isinstance(item, dict):
                        bible["character_states"][character]["inventory"].append(item)
        elif key == "relationships" and isinstance(value, dict):
            # Update relationships dict
            if "relationships" not in bible["character_states"][character]:
                bible["character_states"][character]["relationships"] = {}
            bible["character_states"][character]["relationships"].update(value)
        else:
            bible["character_states"][character][key] = value
    
    save_bible(path, bible)
    print(f"[OK] Updated state for: {character}")


def add_invented_detail(
    path: str, 
    detail: str, 
    chapter: int, 
    scene: int,
    category: str = "general"
) -> None:
    """
    Record an invented detail for author review.
    
    Args:
        path: Project directory
        detail: What was invented
        chapter: Chapter where used
        scene: Scene where used
        category: Type of invention (character, location, power, etc.)
    """
    # Validate category against known list (warning only, still proceeds)
    if category not in INVENTION_CATEGORIES:
        print(f"[WARN] Warning: Unknown category '{category}'. "
              f"Known categories: {', '.join(INVENTION_CATEGORIES)}")

    bible = load_bible(path)
    
    bible["invented_details"].append({
        "detail": detail,
        "category": category,
        "chapter": chapter,
        "scene": scene,
        "reviewed": False,
        "approved": None,
        "added": datetime.now().isoformat()
    })
    
    save_bible(path, bible)
    print(f"[WARN] Flagged invented detail: {detail[:50]}...")


def approve_invention(path: str, detail_index: int, approved: bool = True) -> None:
    """
    Mark an invented detail as reviewed.
    
    Args:
        path: Project directory
        detail_index: Index in invented_details list
        approved: Whether to approve or reject
    """
    bible = load_bible(path)
    
    if detail_index < len(bible["invented_details"]):
        bible["invented_details"][detail_index]["reviewed"] = True
        bible["invented_details"][detail_index]["approved"] = approved
        
        if approved:
            # Add to established facts
            detail = bible["invented_details"][detail_index]["detail"]
            ch = bible["invented_details"][detail_index]["chapter"]
            bible["established_facts"].append({
                "fact": detail,
                "established_in": f"ch{ch}",
                "added": datetime.now().isoformat(),
                "was_invented": True
            })
        
        save_bible(path, bible)
        status = "approved" if approved else "rejected"
        print(f"[OK] Invention {detail_index} {status}")
    else:
        print(f"[WARN] Invalid invention index: {detail_index}")


def reject_invention(path: str, detail_index: int) -> bool:
    """
    Reject an invented detail and remove it from the list entirely.

    Unlike approve_invention(approved=False) which marks the detail as
    rejected but keeps it in the list, this function completely removes
    the rejected detail from invented_details.

    Args:
        path: Project directory
        detail_index: Index in invented_details list (0-based, excluding template entries)

    Returns:
        True if invention was rejected and removed, False otherwise
    """
    bible = load_bible(path)

    # Filter out template/comment entries for accurate indexing
    invented = bible.get("invented_details", [])
    real_invented = [(i, d) for i, d in enumerate(invented) if "_comment" not in d]

    if detail_index < 0 or detail_index >= len(real_invented):
        print(f"[WARN] Invalid invention index: {detail_index}. "
              f"Valid range: 0-{len(real_invented) - 1}")
        return False

    original_index, removed_detail = real_invented[detail_index]

    # Remove from the list
    del bible["invented_details"][original_index]

    save_bible(path, bible)
    print(f"[OK] Rejected and removed invention: {removed_detail.get('detail', '')[:50]}...")
    return True


def update_timeline(path: str, chapter: int, timepoint: str) -> None:
    """
    Record when a chapter takes place.

    For consistency, use a standard format like:
    - "Year 1, Spring" or "Year 1, Day 15"
    - "Day 1, Morning" or "Day 1, Evening"
    - "Same day as Ch X" for continuity

    Args:
        path: Project directory
        chapter: Chapter number
        timepoint: When this chapter occurs (e.g., "Year 1, Spring")
    """
    bible = load_bible(path)

    # Check for format consistency with existing timeline entries
    existing_entries = [v for k, v in bible.get("timeline", {}).items()
                       if not k.startswith("_")]
    if existing_entries:
        # Simple heuristic: check if new format starts similarly to existing
        existing_prefixes = set()
        for entry in existing_entries:
            if entry and len(entry) > 3:
                # Extract first word pattern (e.g., "Year", "Day", "Month")
                first_word = entry.split()[0] if entry.split() else ""
                if first_word:
                    existing_prefixes.add(first_word.lower())

        new_first_word = timepoint.split()[0].lower() if timepoint.split() else ""
        if existing_prefixes and new_first_word and new_first_word not in existing_prefixes:
            if "same" not in new_first_word:  # Allow "Same day" references
                print(f"[NOTE] Timeline format differs from existing entries.")
                print(f"   Existing use: {', '.join(existing_prefixes)}")
                print(f"   Consider using consistent format for easier tracking.")

    bible["timeline"][str(chapter)] = timepoint

    save_bible(path, bible)
    print(f"[OK] Timeline: Chapter {chapter} -> {timepoint}")


def add_location(path: str, name: str, description: str, first_seen: int) -> None:
    """
    Record a location that has been visited.
    
    Args:
        path: Project directory
        name: Location name
        description: Brief description
        first_seen: Chapter where first described
    """
    bible = load_bible(path)
    
    bible["locations"][name] = {
        "description": description,
        "first_seen": f"ch{first_seen}",
        "added": datetime.now().isoformat()
    }
    
    save_bible(path, bible)
    print(f"[OK] Added location: {name}")


def add_mercy_act(
    path: str,
    character: str,
    recipient: str,
    description: str,
    chapter: int,
    scene: int
) -> None:
    """
    Record a mercy act shown by a character.

    Args:
        path: Project directory
        character: Character who showed mercy
        recipient: Who received mercy
        description: What mercy was shown
        chapter: Chapter where it occurred
        scene: Scene where it occurred
    """
    bible = load_bible(path)

    # Ensure mercy_engine section exists (backward compat)
    if "mercy_engine" not in bible:
        bible["mercy_engine"] = {
            "mercy_acts": [],
            "mercy_refused": [],
            "mercy_balance": 0
        }

    bible["mercy_engine"]["mercy_acts"].append({
        "character": character,
        "recipient": recipient,
        "description": description,
        "chapter": chapter,
        "scene": scene,
        "added": datetime.now().isoformat()
    })

    bible["mercy_engine"]["mercy_balance"] += 1

    save_bible(path, bible)
    print(f"[OK] Recorded mercy act: {character} -> {recipient}")


def record_mercy_refused(
    path: str,
    character: str,
    situation: str,
    chapter: int,
    scene: int,
    consequence: str = None
) -> None:
    """
    Record when a character refused to show mercy.

    Args:
        path: Project directory
        character: Character who refused mercy
        situation: Description of situation
        chapter: Chapter where it occurred
        scene: Scene where it occurred
        consequence: What happened as a result (optional)
    """
    bible = load_bible(path)

    # Ensure mercy_engine section exists (backward compat)
    if "mercy_engine" not in bible:
        bible["mercy_engine"] = {
            "mercy_acts": [],
            "mercy_refused": [],
            "mercy_balance": 0
        }

    bible["mercy_engine"]["mercy_refused"].append({
        "character": character,
        "situation": situation,
        "chapter": chapter,
        "scene": scene,
        "consequence": consequence,
        "added": datetime.now().isoformat()
    })

    bible["mercy_engine"]["mercy_balance"] -= 1

    save_bible(path, bible)
    print(f"[WARN] Recorded mercy refused: {character} in ch{chapter}.{scene}")


def get_mercy_status(path: str) -> dict:
    """
    Get the current mercy engine status.

    Args:
        path: Project directory

    Returns:
        Dict with mercy_acts, mercy_refused, and mercy_balance
    """
    bible = load_bible(path)

    if "mercy_engine" not in bible:
        return {
            "mercy_acts": [],
            "mercy_refused": [],
            "mercy_balance": 0,
            "exists": False
        }

    return {
        "mercy_acts": bible["mercy_engine"]["mercy_acts"],
        "mercy_refused": bible["mercy_engine"]["mercy_refused"],
        "mercy_balance": bible["mercy_engine"]["mercy_balance"],
        "exists": True
    }


def update_relationship(
    path: str,
    char_a: str,
    char_b: str,
    status: str,
    chapter: int = None
) -> None:
    """
    Update the relationship between two characters.

    Updates the top-level 'relationships' section in the story bible,
    which tracks pair-based relationships with history.

    Args:
        path: Project directory
        char_a: First character name
        char_b: Second character name
        status: Current relationship status/description
        chapter: Chapter where this change occurred (for history tracking)
    """
    bible = load_bible(path)

    # Create canonical key (alphabetically sorted for consistency)
    key = f"{min(char_a, char_b)}:{max(char_a, char_b)}"

    # Initialize relationships section if needed
    if "relationships" not in bible:
        bible["relationships"] = {}

    # Get or create relationship entry
    if key not in bible["relationships"]:
        bible["relationships"][key] = {
            "status": status,
            "history": []
        }
    else:
        # Record previous status to history before updating
        old_status = bible["relationships"][key].get("status", "")
        if chapter and old_status:
            bible["relationships"][key]["history"].append({
                "chapter": chapter,
                "previous_status": old_status,
                "changed_to": status,
                "timestamp": datetime.now().isoformat()
            })
        bible["relationships"][key]["status"] = status

    bible["relationships"][key]["updated"] = datetime.now().isoformat()

    save_bible(path, bible)
    print(f"[OK] Updated relationship: {char_a} <-> {char_b}")


def get_continuity_report(path: str) -> str:
    """
    Generate a continuity report for review.

    Args:
        path: Project directory

    Returns:
        Formatted report string
    """
    try:
        bible = load_bible(path)
    except (FileNotFoundError, ValueError) as e:
        return f"# Continuity Report\n\nError loading story bible: {e}"

    try:
        report = []
        report.append("# Continuity Report\n")
        report.append(f"Generated: {datetime.now().isoformat()}\n")

        # Progress - use defensive .get() calls
        progress = bible.get("progress", {})
        report.append(f"\n## Progress")
        report.append(f"- Current position: Chapter {progress.get('current_chapter', '?')}, Scene {progress.get('current_scene', '?')}")
        total_words = progress.get('total_words', 0)
        report.append(f"- Words written: {total_words:,}")
        report.append(f"- Chapters complete: {progress.get('chapters_complete', 0)}")

        # Unresolved foreshadowing (filter out _comment template entries)
        foreshadowing = bible.get("foreshadowing", {})
        planted = foreshadowing.get("planted", [])
        unresolved = [
            p for p in planted
            if not p.get("paid_off") and "_comment" not in p
        ]
        if unresolved:
            report.append(f"\n## Unresolved Foreshadowing ({len(unresolved)})")
            for plant in unresolved:
                report.append(f"- {plant.get('thread', 'Unknown')} (planted: {plant.get('planted_in', '?')})")

        # Unreviewed inventions (filter out _comment template entries)
        invented_details = bible.get("invented_details", [])
        unreviewed = [
            d for d in invented_details
            if not d.get("reviewed") and "_comment" not in d
        ]
        if unreviewed:
            report.append(f"\n## Unreviewed Inventions ({len(unreviewed)})")
            for i, detail in enumerate(unreviewed):
                report.append(f"- [{i}] {detail.get('detail', 'Unknown')} (ch{detail.get('chapter', '?')}.{detail.get('scene', '?')})")

        # Character states (filter out _example keys from template)
        report.append(f"\n## Current Character States")
        character_states = bible.get("character_states", {})
        for char, state in _filter_examples(character_states).items():
            report.append(f"\n### {char}")
            if state.get("location"):
                report.append(f"- Location: {state['location']}")
            if state.get("emotional_state"):
                report.append(f"- Emotional state: {state['emotional_state']}")
            if state.get("knows"):
                # Handle structured knows format
                knows_list = _normalize_knows(state['knows'])
                knows_display = []
                for k in knows_list[-5:]:  # Last 5 items
                    if isinstance(k, dict) and "fact" in k:
                        fact = k["fact"]
                        if k.get("learned_in"):
                            knows_display.append(f"{fact} ({k['learned_in']})")
                        else:
                            knows_display.append(fact)
                    else:
                        knows_display.append(str(k))
                report.append(f"- Knows: {', '.join(knows_display)}")
            if state.get("inventory"):
                inv_items = [i.get("item", str(i)) if isinstance(i, dict) else str(i)
                             for i in state["inventory"]]
                report.append(f"- Inventory: {', '.join(inv_items)}")

        # Mercy Engine status
        if "mercy_engine" in bible:
            me = bible["mercy_engine"]
            report.append(f"\n## Mercy Engine")
            report.append(f"- Balance: {me.get('mercy_balance', 0)}")
            report.append(f"- Acts of mercy: {len(me.get('mercy_acts', []))}")
            report.append(f"- Mercy refused: {len(me.get('mercy_refused', []))}")

        return "\n".join(report)

    except Exception as e:
        return f"# Continuity Report\n\nError generating report: {e}"


def update_chapter(path: str, chapter: int, summary: str = None, timeline: str = None) -> None:
    """
    Perform a comprehensive chapter update in the story bible.

    This is the function referenced by SKILL.md for post-chapter updates.
    It updates chapter summary, timeline, and triggers any chapter-end processing.
    Also syncs with draft-state.json to prevent state desynchronization.

    Args:
        path: Project directory
        chapter: Chapter number
        summary: Chapter summary (100-200 words)
        timeline: When this chapter occurs (e.g., "Year 1, Spring")
    """
    bible = load_bible(path)

    chapter_key = str(chapter)

    # Ensure chapter exists in bible
    if chapter_key not in bible["chapters"]:
        bible["chapters"][chapter_key] = {
            "title": f"Chapter {chapter}",
            "word_count": 0,
            "scenes": {},
            "summary": "",
            "final_state": {},
            "completed": False,
            "completed_at": None
        }

    # Update summary if provided
    if summary:
        bible["chapters"][chapter_key]["summary"] = summary
        print(f"[OK] Updated chapter {chapter} summary")

    # Update timeline if provided
    if timeline:
        bible["timeline"][chapter_key] = timeline
        print(f"[OK] Updated timeline: Chapter {chapter} -> {timeline}")

    # Generate chapter status
    ch_data = bible["chapters"][chapter_key]
    scene_count = len(ch_data.get("scenes", {}))
    word_count = ch_data.get("word_count", 0)

    print(f"\nChapter {chapter} Status:")
    print(f"   Scenes: {scene_count}")
    print(f"   Words: {word_count:,}")
    print(f"   Completed: {'Yes' if ch_data.get('completed') else 'No'}")

    if ch_data.get("summary"):
        print(f"   Summary: {ch_data['summary'][:80]}...")

    save_bible(path, bible)

    # === SYNC WITH DRAFT-STATE.JSON ===
    # Keep draft-state.json synchronized with story-bible.json
    _sync_draft_state(path, bible)


def get_chapter_status(path: str, chapter: int) -> dict:
    """
    Get the current status of a specific chapter.

    Args:
        path: Project directory
        chapter: Chapter number

    Returns:
        Dict with chapter status information
    """
    bible = load_bible(path)
    chapter_key = str(chapter)

    if chapter_key not in bible["chapters"]:
        return {"exists": False, "chapter": chapter}

    ch_data = bible["chapters"][chapter_key]

    return {
        "exists": True,
        "chapter": chapter,
        "title": ch_data.get("title", f"Chapter {chapter}"),
        "word_count": ch_data.get("word_count", 0),
        "scene_count": len(ch_data.get("scenes", {})),
        "completed": ch_data.get("completed", False),
        "summary": ch_data.get("summary", ""),
        "timeline": bible.get("timeline", {}).get(chapter_key, "Not set")
    }


if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("Usage:")
        print("  update_story_bible.py <path> --chapter <num> [--summary <text>] [--timeline <text>]")
        print("  update_story_bible.py <path> --fact <chapter> <scene> <fact>")
        print("  update_story_bible.py <path> --plant <chapter> <scene> <thread>")
        print("  update_story_bible.py <path> --payoff <chapter> <scene> <thread>")
        print("  update_story_bible.py <path> --character <name> <key> <value> [--chapter <num>]")
        print("  update_story_bible.py <path> --character <name> --json '{\"key\": \"value\"}' [--chapter <num>]")
        print("  update_story_bible.py <path> --invented <chapter> <scene> <detail>")
        print("  update_story_bible.py <path> --report")
        print("  update_story_bible.py <path> --status <chapter>")
        print("  update_story_bible.py <path> --mark-review-complete [chapter]")
        print("  update_story_bible.py <path> --relationship <char_a> <char_b> <status> [--chapter <num>]")
        print("  update_story_bible.py <path> --location <name> <description> <first_seen_chapter>")
        print("  update_story_bible.py <path> --mercy-act <character> <recipient> <description> <chapter> <scene>")
        print("  update_story_bible.py <path> --mercy-refused <character> <situation> <chapter> <scene> [consequence]")
        print("  update_story_bible.py <path> --mercy-status")
        print("  update_story_bible.py <path> --remove-fact <index_or_text>")
        print("  update_story_bible.py <path> --remove-plant <thread_text>")
        print("  update_story_bible.py <path> --reject-invention <index>")
        print("")
        print("Examples:")
        print("  # Update single field:")
        print("  update_story_bible.py ./project --character Sonny location \"the tavern\"")
        print("")
        print("  # Update multiple fields with JSON:")
        print("  update_story_bible.py ./project --character Sonny --json '{\"location\": \"tavern\", \"emotional_state\": \"anxious\"}'")
        print("")
        print("  # Update with history tracking (records state change at chapter):")
        print("  update_story_bible.py ./project --character Sonny location \"forest\" --chapter 5")
        sys.exit(1)

    path = sys.argv[1]
    command = sys.argv[2]

    if command == "--chapter" and len(sys.argv) >= 4:
        chapter_num = int(sys.argv[3])
        # Parse optional arguments
        summary = None
        timeline = None
        i = 4
        while i < len(sys.argv):
            if sys.argv[i] == "--summary" and i + 1 < len(sys.argv):
                summary = sys.argv[i + 1]
                i += 2
            elif sys.argv[i] == "--timeline" and i + 1 < len(sys.argv):
                timeline = sys.argv[i + 1]
                i += 2
            else:
                i += 1
        update_chapter(path, chapter_num, summary, timeline)
    elif command == "--status" and len(sys.argv) >= 4:
        status = get_chapter_status(path, int(sys.argv[3]))
        if status["exists"]:
            print(f"Chapter {status['chapter']}: {status['title']}")
            print(f"  Words: {status['word_count']:,}")
            print(f"  Scenes: {status['scene_count']}")
            print(f"  Completed: {'Yes' if status['completed'] else 'No'}")
            print(f"  Timeline: {status['timeline']}")
            if status['summary']:
                print(f"  Summary: {status['summary'][:100]}...")
        else:
            print(f"Chapter {status['chapter']} not found in story bible")
    elif command == "--fact" and len(sys.argv) >= 6:
        add_established_fact(path, sys.argv[5], int(sys.argv[3]), int(sys.argv[4]))
    elif command == "--plant" and len(sys.argv) >= 6:
        add_foreshadowing_plant(path, sys.argv[5], int(sys.argv[3]), int(sys.argv[4]))
    elif command == "--payoff" and len(sys.argv) >= 6:
        record_payoff(path, sys.argv[5], int(sys.argv[3]), int(sys.argv[4]))
    elif command == "--character" and len(sys.argv) >= 4:
        character_name = sys.argv[3]
        updates = {}
        chapter_for_history = None

        # Check if using --json mode or key-value mode
        if len(sys.argv) >= 5 and sys.argv[4] == "--json":
            # JSON mode: --character <name> --json '{"key": "value"}'
            if len(sys.argv) >= 6:
                try:
                    updates = json.loads(sys.argv[5])
                except json.JSONDecodeError as e:
                    print(f"Error: Invalid JSON: {e}")
                    sys.exit(1)
                # Check for --chapter after JSON
                if "--chapter" in sys.argv[6:]:
                    idx = sys.argv.index("--chapter", 6)
                    if idx + 1 < len(sys.argv):
                        chapter_for_history = int(sys.argv[idx + 1])
            else:
                print("Error: --json requires a JSON string argument")
                sys.exit(1)
        elif len(sys.argv) >= 6:
            # Key-value mode: --character <name> <key> <value>
            updates = {sys.argv[4]: sys.argv[5]}
            # Check for --chapter after key-value
            if "--chapter" in sys.argv[6:]:
                idx = sys.argv.index("--chapter", 6)
                if idx + 1 < len(sys.argv):
                    chapter_for_history = int(sys.argv[idx + 1])
        else:
            print("Error: --character requires either '<key> <value>' or '--json <json_string>'")
            sys.exit(1)

        update_character_state(path, character_name, updates, chapter_for_history)
    elif command == "--invented" and len(sys.argv) >= 6:
        detail = sys.argv[5]
        chapter = int(sys.argv[3])
        scene = int(sys.argv[4])
        category = "general"
        # Parse optional --category flag
        if "--category" in sys.argv[6:]:
            idx = sys.argv.index("--category", 6)
            if idx + 1 < len(sys.argv):
                category = sys.argv[idx + 1]
        add_invented_detail(path, detail, chapter, scene, category)
    elif command == "--report":
        print(get_continuity_report(path))
    elif command == "--mark-review-complete":
        chapters = int(sys.argv[3]) if len(sys.argv) >= 4 else None
        mark_review_complete(path, chapters)
    elif command == "--relationship" and len(sys.argv) >= 6:
        char_a = sys.argv[3]
        char_b = sys.argv[4]
        status = sys.argv[5]
        chapter_num = None
        if "--chapter" in sys.argv[6:]:
            idx = sys.argv.index("--chapter", 6)
            if idx + 1 < len(sys.argv):
                chapter_num = int(sys.argv[idx + 1])
        update_relationship(path, char_a, char_b, status, chapter_num)
    elif command == "--location" and len(sys.argv) >= 6:
        name = sys.argv[3]
        description = sys.argv[4]
        first_seen = int(sys.argv[5])
        add_location(path, name, description, first_seen)
    elif command == "--mercy-act" and len(sys.argv) >= 8:
        character = sys.argv[3]
        recipient = sys.argv[4]
        description = sys.argv[5]
        chapter_num = int(sys.argv[6])
        scene_num = int(sys.argv[7])
        add_mercy_act(path, character, recipient, description, chapter_num, scene_num)
    elif command == "--mercy-refused" and len(sys.argv) >= 7:
        character = sys.argv[3]
        situation = sys.argv[4]
        chapter_num = int(sys.argv[5])
        scene_num = int(sys.argv[6])
        consequence = sys.argv[7] if len(sys.argv) >= 8 else None
        record_mercy_refused(path, character, situation, chapter_num, scene_num, consequence)
    elif command == "--mercy-status":
        status = get_mercy_status(path)
        if status["exists"]:
            print(f"Mercy Balance: {status['mercy_balance']}")
            print(f"Acts of Mercy: {len(status['mercy_acts'])}")
            print(f"Mercy Refused: {len(status['mercy_refused'])}")
            if status['mercy_acts']:
                print("\nRecent mercy acts:")
                for act in status['mercy_acts'][-3:]:
                    print(f"  - {act['character']} -> {act['recipient']} (ch{act['chapter']})")
        else:
            print("No mercy_engine section found in story bible")
    elif command == "--remove-fact" and len(sys.argv) >= 4:
        arg = sys.argv[3]
        # Try to parse as integer (index), otherwise treat as text match
        try:
            index = int(arg)
            remove_established_fact(path, index=index)
        except ValueError:
            remove_established_fact(path, match_text=arg)
    elif command == "--remove-plant" and len(sys.argv) >= 4:
        remove_foreshadowing_plant(path, sys.argv[3])
    elif command == "--reject-invention" and len(sys.argv) >= 4:
        reject_invention(path, int(sys.argv[3]))
    else:
        print(f"Unknown command or missing arguments: {command}")
        sys.exit(1)
