# 任务模式 - Claude Code 的操作工作流

**为常见任务类型提供结构化工作流**  
**最后更新：** 2025-11-12

---

## 🎯 目的

本文件为不同类型的任务提供分步工作流，确保一致性、质量，以及与项目愿景对齐。每种任务类型都包含清晰的推理链与执行顺序。

---

## 📋 任务类别

```
🧩 功能实现
🐛 Bug 修复
🧠 重构 / 性能优化
🧾 文档更新
🔄 部署 / CI/CD 任务
🔍 调研 / 代码探索
🧪 测试
🔐 安全增强
```

---

## 🧩 功能实现

### 适用场景
- 增加新功能
- 构建新的 API 接口
- 创建新的 UI 组件
- 实现业务逻辑

### 开始前检查清单

**开始任何功能实现前，请确认：**

```yaml
✅ 理解核查：
□ 我清楚要构建什么（具体需求）
□ 我清楚为何要构建（业务价值、战略目标）
□ 我清楚谁会使用（用户角色）
□ 我清楚放在哪（对应的模块/组件）
□ 我清楚时间要求（紧急程度）

✅ 上下文核对：
□ 是否符合 PROJECT_VISION.md 的战略目标？
□ 属于 Zoho 同步逻辑还是本地 ERP 逻辑？
□ 当前 Zoho 迁移阶段？（阶段 1：只读）
□ 当前环境？（开发 = 随时可部署）
□ 当前 git 分支？（应为 develop 或 feature/*）

✅ 模块影响评估：
□ 影响哪个模块？
  - 销售（订单、发票、支付）
  - 库存（产品、库存、仓库）
  - 人力（用户、销售人员、佣金）
  - 客户（批发客户、合作销售）
  - 报表（分析、仪表盘）
  - TDS 核心（Zoho 同步操作）

✅ 阿拉伯语与本地化：
□ 是否包含面向用户的文本？（是 = 需要阿拉伯语）
□ 是否创建/修改数据库字段？（需添加 name_ar、description_ar）
□ UI 是否需要 RTL 布局支持？（阿拉伯语场景下始终需要）
□ 此功能的主要语言？（阿拉伯语）

✅ 认证与鉴权：
□ 谁可访问此功能？（所有用户、特定角色、仅管理员）
□ 是否修改数据？（是 = 需要认证）
□ 是否访问敏感数据？（是 = 需要 RBAC 检查）
□ 允许哪些角色？（admin、manager、salesperson、client）

✅ 集成点：
□ 是否与 Zoho 交互？（是 = 必须使用 TDS Core）
□ 是否需要实时数据？（或可接受缓存）
□ 是否与现有功能集成？（哪一些）
□ 是否需要移动端改动？（8 个应用中的哪一些）

✅ 性能与规模：
□ 是否需支持 500+ 客户？
□ 是否需支持 2,218+ 产品？
□ 是否需支持每天 30+ 订单？
□ 是否需要分页？（> 100 条记录 = 是）
□ 是否需要数据库索引？（大表查询）

✅ 依赖核查：
□ 是否已有类似功能？（先搜索代码库）
□ 可复用的现有代码？（services、models、schemas）
□ 需要新增外部依赖？（Python 包）
□ 是否需要数据库变更？（迁移）
```

**决策点：仅在所有检查通过后继续。**

### 分步工作流

#### 第 1 步：上下文与对齐（必做）
```yaml
Actions:
  1. 阅读 PROJECT_VISION.md
     - 确认功能与业务目标对齐
     - 检查当前 Zoho 迁移阶段限制
     - 校验规模要求（500+ 客户、2,218+ 产品）

  2. 阅读 ARCHITECTURE_RULES.md
     - 验证技术栈兼容性
     - 检查需遵循的设计模式
     - 回顾命名约定

  3. 澄清问题
     - 谁会使用此功能？（用户角色）
     - 业务预期是什么？
     - 是否有性能要求？
     - 面向移动端、Web，或两者？
     - 需要阿拉伯语支持吗？（通常需要，确认 UI 要求）

Decision Point: 对齐确认后再继续
```

#### 第 2 步：搜索现有代码（必做）
```yaml
Actions:
  1. 搜索相似功能
     - 使用 Grep 进行代码库搜索
     - 检查 /app/routers/、/app/services/、/scripts/
     - 查找已有模式

  2. 评估：增强 vs 新建
     - 若已有类似：增强现有代码
     - 若为全新：按既有模式新建

Example Commands:
  grep -r "commission" app/services/
  grep -r "product.*list" app/routers/
  find . -name "*wholesale*" -type f
```

#### 第 3 步：规划与设计
```yaml
Actions:
  1. 若复杂（≥3 步），创建 TodoWrite 列表
  2. 设计 API 契约（若后端）
     - 路径：/api/v1/resource
     - 请求/响应 schema
     - 认证要求
     - RBAC 角色允许

  3. 设计数据库变更（如需）
     - 新表或新列
     - 所需索引
     - 迁移计划

  4. 设计 UI 流程（若前端）
     - 页面布局
     - 用户交互
     - 阿拉伯语 RTL 考量
```

#### 第 4 步：实现
```yaml
Backend（FastAPI）：
  1. 在 app/schemas/ 创建 Pydantic schema
  2. 在 app/models/ 创建/更新 SQLAlchemy 模型
  3. 在 app/services/ 创建服务层
  4. 在 app/routers/ 创建路由/接口
  5. 添加输入校验
  6. 添加错误处理
  7. 添加认证/鉴权
  8. 包含阿拉伯语字段（name_ar、description_ar）
  9. 添加带业务背景的文档字符串

Frontend（React/Flutter）：
  1. 创建组件/控件
  2. 实现状态管理
  3. 集成 API
  4. 实现阿拉伯语 RTL 布局
  5. 添加加载态
  6. 添加错误处理
  7. 添加校验
  8. 保证移动端响应式

代码质量：
  - 遵循命名约定
  - 添加类型标注（Python/TypeScript）
  - 编写清晰文档字符串
  - 为复杂业务逻辑添加注释
  - 不要硬编码（使用常量）
```

#### 第 5 步：测试
```yaml
Actions：
  1. 本地测试
     - 正常路径（预期用法）
     - 错误场景
     - 边界场景
     - 阿拉伯语文本输入
     - 大数据量（分页）

  2. 集成测试
     - API 正常响应
     - 数据库操作正常
     - 前端正确展示数据

  3. 真实场景测试
     - 使用接近生产的数据量
     - 按真实用户工作流测试
```

#### 第 6 步：文档
```yaml
Actions：
  1. API 文档（FastAPI 自动生成）
     - 确认 OpenAPI 文档清晰
     - 添加详细描述

  2. 代码文档
     - 函数/类文档字符串
     - 复杂逻辑的内联注释
     - 解释业务背景

  3. 更新 CHANGELOG.md（若为重要功能）
     - 添加到 “Unreleased” 部分
     - 描述新增内容
```

#### 第 7 步：部署
```yaml
Actions：
  1. 使用清晰信息提交
     - git add .
     - git commit -m "feat: add wholesale client credit limit checking"

  2. 推送到 develop（预发布）
     - git push origin develop

  3. 监控部署
     - gh run watch <run-id>

  4. 在预发布环境验证
     - 在 staging.erp.tsh.sale 测试功能
     - 获取 Khaleel 的确认

  5. 创建生产 PR
     - gh pr create --base main --head develop
     - 附上测试结果

Decision Point：仅在预发布验证通过后标记完成
```

### 质量门（必须通过）
```yaml
✅ 与 PROJECT_VISION.md 对齐
✅ 遵循 ARCHITECTURE_RULES.md 模式
✅ 已搜索现有相似代码
✅ 包含阿拉伯语支持（若面向用户）
✅ 已实现输入校验
✅ 已实现错误处理
✅ 已进行认证/鉴权检查
✅ 本地测试包含边界场景
✅ 已撰写文档（docstrings + 注释）
✅ 已在预发布环境验证
```

---

## 🐛 Bug 修复

### 适用场景
- 某处功能损坏
- 来自用户的错误报告
- 测试失败
- 非预期行为

### 开始前检查清单

**开始任何 Bug 修复前，请确认：**

```yaml
✅ Bug 验证：
□ 我能复现 Bug 吗？（具体步骤）
□ 具体错误或非预期行为是什么？
□ 是稳定复现还是偶发？
□ 哪个环境？（本地、预发布、生产）
□ 何时开始出现？（最近部署？一直存在？）

✅ 严重性评估：
□ 影响级别：
  - 严重：生产不可用、数据损坏、安全事故
  - 高：核心功能损坏，影响多数用户
  - 中：部分功能损坏，影响部分用户
  - 低：小问题，有替代方案

□ 紧急程度：
  - 立即：立刻修复（生产不可用时启用紧急模式）
  - 紧急：数小时内修复（影响业务运营）
  - 正常：数日内修复（计划内工作）
  - 低：有空再修（锦上添花）

✅ 用户影响：
□ 受影响用户数？（全部 500+ 客户或特定用户）
□ 受影响的角色？（admin、manager、salesperson、client）
□ 受阻的工作流？（创建订单、支付、报表）
□ 是否可绕过？（是 = 降低优先级）

✅ 根因假设：
□ 最近代码变更？（检查最近 7 天 git log）
□ 最近部署？（检查 GitHub Actions）
□ 数据库问题？（连接、迁移、数据一致性）
□ 外部服务问题？（Zoho、VPS、网络）
□ 环境问题？（环境变量、配置）
□ 数据特定？（仅在特定数据下发生）

✅ 证据收集：
□ 错误日志？（后端日志、TDS Core 日志）
□ 堆栈信息？（完整错误细节）
□ 最近提交审查？（git log --since="3 days ago"）
□ 数据库状态？（查询验证数据）
□ 用户操作？（错误前的操作）

✅ 修复策略：
□ 修根因还是止症状？（优先修根因）
□ 快速修复还是规范修复？（严重时快速，其它情况规范）
□ 可回滚？（若因最近部署导致）
□ 现在修复安全吗？（或等待维护窗口）
□ 是否有测试环境？（预发布验证修复）
```

**决策点：在动手修复前必须理解根因。**

### 分步工作流

#### 第 1 步：复现与诊断
```yaml
Actions：
  1. 复现 Bug
     - 按确切步骤触发
     - 记录环境（本地、预发布、生产）
     - 捕获错误信息

  2. 收集信息
     - 查看日志（journalctl、浏览器控制台）
     - 检查数据库状态
     - 检查近期代码变更（git log）

  3. 确认根因
     - 使用调试工具
     - 必要时添加临时日志
     - 理解“为什么”，不仅是“是什么”

Example：
  “错误发生在为不存在的 customer_id 创建订单时。
   根因：客户被删除，但创建订单前未校验客户存在。”
```

#### 第 2 步：影响评估
```yaml
Questions：
  1. 严重性
     - 严重：生产不可用、数据丢失
     - 高：核心功能损坏
     - 中：次要功能损坏
     - 低：外观问题

  2. 范围
     - 受影响用户数？
     - 受影响功能点？
     - 是否影响数据完整性？

  3. 紧急性
     - 立即：生产严重问题
     - 高：当天应修复
     - 中：下个迭代修复
     - 低：进入待办

Decision：据此排序优先级
```

#### 第 3 步：设计修复
```yaml
Actions：
  1. 确定修复方式
     - 临时补丁 vs 规范方案
     - 是否有副作用？
     - 是否需要数据库迁移？

  2. 检查类似问题
     - 搜索代码库中的相同模式
     - 如有需要，多处一并修复

  3. 计划预防措施
     - 增加校验避免复发
     - 添加测试用例
     - 必要时更新文档
```

#### 第 4 步：实施修复
```yaml
Actions：
  1. 编写修复
     - 最小化改动（不重构无关代码）
     - 清晰聚焦的解决方案
     - 添加注释解释修复点

  2. 增加预防
     - 输入校验
     - 错误处理
     - 数据库约束
     - 测试用例

  3. 验证修复有效
     - 测试最初失败的场景
     - 测试相关场景
     - 确保无回归
```

#### 第 5 步：部署与验证
```yaml
严重 Bug：
  1. 从 main 创建 hotfix 分支
     - git checkout main
     - git checkout -b hotfix/critical-bug-name

  2. 实施并测试修复

  3. 先推送到 develop（快速预发布测试）
     - git push origin develop
     - 在预发布快速验证

  4. 加速创建到 main 的 PR
     - gh pr create --base main --head develop --label "hotfix"
     - 快速审批

  5. 紧密监控生产环境
     - 验证修复已部署
     - 监控 15-30 分钟
     - 检查错误日志

非严重 Bug：
  1. 按常规流程：develop → staging → main
  2. 在预发布充分测试
  3. 正常 PR 流程
```

### 质量门（必须通过）
```yaml
✅ 已识别并理解根因
✅ 使用原始失败场景测试修复
✅ 无回归（其他功能仍正常）
✅ 已添加预防机制（校验/测试）
✅ 精确评估影响
✅ 生产前已在预发布验证
```

---

## 🧠 重构 / 优化

### 适用场景
- 代码可读性差或难以理解
- 性能问题
- 技术债务
- 重复代码

### 分步工作流

#### 第 1 步：先度量（必做）
```yaml
Actions：
  1. 建立基线
     - 当前性能指标
     - 当前代码复杂度
     - 当前测试覆盖率

  2. 明确问题
     - 具体问题是什么？
     - 如何确认它是问题？
     - 影响是什么？

Example：
  ❌ 不佳：“优化产品接口”
  ✅ 良好：“产品接口在 2,218 个产品时耗时 2.3s，
           目标 <500ms。当前查询存在 N+1 选择。”

Rule：没有度量就不要优化
```

#### 第 2 步：规划重构
```yaml
Actions：
  1. 定义目标
     - 会改善什么？
     - 如何度量成功？
     - 目标指标是什么？

  2. 评估风险
     - 影响多少代码？
     - 能否拆分为更小的改动？
     - 是否需要特性开关？

  3. 测试策略
     - 现有测试应保持通过
     - 若覆盖不足应新增测试
     - 如何验证无回归？
```

#### 第 3 步：实施改动
```yaml
Actions：
  1. 增量式改动
     - 小步快跑，可测试
     - 频繁提交
     - 不要重构与加功能同时进行

  2. 保持行为一致
     - 相同输入 → 相同输出
     - 不破坏 API 契约
     - 保持向后兼容

  3. 提升清晰度
     - 更好的变量命名
     - 更清晰的函数拆分
     - 去重重复代码
     - 为复杂逻辑添加注释
```

#### 第 4 步：事后度量（必做）
```yaml
Actions：
  1. 验证改进
     - 与基线使用相同指标度量
     - 是否达成目标？
     - 是否有副作用？

  2. 记录结果
     - 之前：X
     - 之后：Y
     - 改进：Z%

Example：
  “优化产品列表接口：
   - 之前：2,218 产品耗时 2.3s
   - 之后：使用分页+索引耗时 180ms
   - 改进：提升 92%
   - 方法：添加分页，给 category_id 与 is_active 建索引”
```

### 质量门（必须通过）
```yaml
✅ 改动前已度量基线
✅ 目标明确且有指标
✅ 所有现有测试仍通过
✅ 改进已度量并记录
✅ 无破坏性改动
✅ 重构后代码更清晰/更简单
```

---

## 🧾 文档更新

### 适用场景
- 添加/更新项目文档
- 更新 .claude/ 系统文件
- 编写指南或 README
- 更新 API 文档

### 分步工作流

#### 第 1 步：识别需求
```yaml
Questions：
  1. 哪些内容过时或缺失？
  2. 目标读者是谁？
     - 开发者（当前或未来）
     - AI 代理（如本助手）
     - 终端用户
     - 系统管理员

  3. 目标是什么？
     - 入门
     - 故障排查
     - 参考
     - 教程
```

#### 第 2 步：研究内容
```yaml
Actions：
  1. 收集准确信息
     - 检查当前代码库
     - 验证配置
     - 测试流程
     - 检查 URL/链接

  2. 回顾现有文档
     - 已记录了什么？
     - 哪些可以增强？
     - 哪些存在矛盾？
```

#### 第 3 步：撰写内容
```yaml
结构：
  1. 标题与目的
  2. 先决条件（如适用）
  3. 分步说明
  4. 示例
  5. 故障排查（如适用）
  6. 相关文档链接

风格：
  - 语言清晰简洁
  - 命令/代码使用代码块
  - 列表使用项目符号
  - 使用标题分区
  - 如有帮助添加图示

针对 .claude/ 文件：
  - 遵循现有格式
  - 更新 “Last Updated” 日期
  - 保持结构一致
  - 与其他文件交叉引用
```

#### 第 4 步：核验准确性
```yaml
Actions：
  1. 测试所有命令/流程
     - 复制粘贴并实际运行
     - 验证与文档一致
     - 检查所有链接有效

  2. 复查清晰度
     - 新开发者可理解吗？
     - 是否存在歧义？
     - 示例是否有帮助？

  3. 检查一致性
     - 与其他文档一致？
     - 术语是否统一？
     - 是否无矛盾？
```

#### 第 5 步：更新相关文档
```yaml
Actions：
  1. 更新交叉引用
     - 链接至新文档
     - 更新索引或目录
     - 更新 CHANGELOG_AI.md（若为 .claude/ 文件）

  2. 归档过时文档
     - 移至 /archived/（若过时）
     - 更新指向旧文档的链接
```

### 质量门（必须通过）
```yaml
✅ 所有命令/流程已测试
✅ 所有链接已验证
✅ 文字清晰简洁
✅ 适配目标读者
✅ 交叉引用已更新
✅ “Last Updated” 日期为当前
```

---

## 🔄 部署 / CI/CD 任务

### 适用场景
- 部署到预发布或生产
- 更新 CI/CD 工作流
- 基础设施变更
- 环境配置变更

### 分步工作流

#### 第 1 步：部署前检查清单
```yaml
必须核验：
  1. 所有组件就绪
     - [ ] 后端 API
     - [ ] ERP 管理前端
     - [ ] 消费者应用（Flutter Web）
     - [ ] TDS Core Worker
     - [ ] TDS Dashboard

  2. 测试通过
     - [ ] 本地测试通过
     - [ ] CI 测试通过
     - [ ] Lint 通过

  3. 依赖检查
     - [ ] requirements.txt 最新
     - [ ] package.json 最新
     - [ ] pubspec.yaml 最新

  4. 环境变量
     - [ ] VPS 上 .env.production 最新
     - [ ] GitHub Secrets 已配置
     - [ ] 代码中不包含密钥

  5. 数据库迁移
     - [ ] 已创建迁移
     - [ ] 已在本地测试迁移
     - [ ] 备份方案就绪
```

#### 第 2 步：先部署到预发布
```yaml
Actions：
  1. 推送到 develop 分支
     - git push origin develop

  2. 监控 GitHub Actions
     - gh run list --limit 5
     - gh run watch <run-id>

  3. 验证预发布部署
     - curl https://staging.erp.tsh.sale/health
     - curl https://staging.consumer.tsh.sale/
     - 手动验证改动

  4. 获取 Khaleel 的批准
     - 展示可工作的功能
     - 确认生产就绪

Decision Point：未经预发布验证不要进入生产
```

#### 第 3 步：生产部署
```yaml
Actions：
  1. 创建 Pull Request
     - gh pr create --base main --head develop
     - 标题：“Deploy: [Feature Name] to Production”
     - 描述：
       * 改动内容
       * 预发布验证结果
       * 注意事项

  2. 评审并合并
     - 代码改动审查
     - Khaleel 批准
     - 合并 PR（触发生产部署）

  3. 监控部署
     - gh run watch <run-id>
     - 关注错误

  4. 验证所有组件
     - 后端：curl https://erp.tsh.sale/health
     - 前端：curl https://erp.tsh.sale/
     - 消费者：curl https://consumer.tsh.sale/
     - TDS Dashboard：curl https://erp.tsh.sale/tds-admin/

  5. 部署后监控
     - 监控 15-30 分钟
     - 检查错误日志
     - 留意用户报告
```

#### 第 4 步：回滚计划（如需）
```yaml
若部署失败：
  1. 评估严重性
     - 生产是否完全不可用？
     - 数据是否有风险？
     - 用户是否仍可工作？

  2. 快速回滚选项
     - VPS 上的蓝绿切换
     - Git revert + 重新部署
     - 从备份恢复（最后手段）

  3. 执行回滚
     - ssh root@167.71.39.50
     - bash /opt/tsh_erp/bin/switch_deployment.sh

  4. 调查并修复
     - 诊断问题原因
     - 在 develop 分支修复
     - 充分测试
     - 重新部署
```

### 质量门（必须通过）
```yaml
✅ 所有组件均已部署（不是只部署一个）
✅ 生产前先验证预发布
✅ 所有健康检查通过
✅ 所有 URL 返回 200 OK
✅ 数据库迁移已成功执行
✅ 日志无错误
✅ 部署已获 Khaleel 批准
```

---

## 🔍 调研 / 研究

### 适用场景
- 理解现有代码
- 研究最佳实践
- 评估技术选型
- 探索代码库结构

### 分步工作流

#### 第 1 步：明确问题
```yaml
Actions：
  1. 澄清要找的内容
     - 具体："如何计算佣金？"
     - 不要模糊："钱是怎么回事？"

  2. 定义成功标准
     - 哪些信息能回答问题？
     - 深度需要到什么程度？
```

#### 第 2 步：搜索策略
```yaml
工具：
  1. Grep（代码搜索）
     - 搜索关键词
     - 查找相似模式

  2. Read（文件阅读）
     - 阅读相关文件
     - 理解上下文

  3. Task（复杂搜索）
     - 使用 Explore 代理深入调查
     - 指定详尽程度

Example：
  “需要找到批发定价的计算方式”

  1. grep -r "wholesale.*price" app/
  2. grep -r "commission" app/services/
  3. 阅读找到的文件
  4. 追踪函数调用
```

#### 第 3 步：记录发现
```yaml
Actions：
  1. 总结发现
     - 清晰解释
     - 文件位置（含行号）
     - 工作机制

  2. 提供示例
     - 实际代码片段
     - 使用示例

  3. 记录问题
     - 发现的 Bug
     - 不一致之处
     - 改进机会

Example Response：
  “批发定价在 app/services/pricing.py:45 计算。

   函数 calculate_wholesale_price() 包含：
   1. 基础产品价格
   2. 量级折扣（>100 单位 = 10%）
   3. 客户表中的客户特定折扣

   当前未考虑 PROJECT_VISION.md 中提到的季节性折扣。”
```

### 质量门（必须通过）
```yaml
✅ 问题定义清晰
✅ 进行了彻底搜索（不仅仅第一个结果）
✅ 发现包含文件路径与行号
✅ 回答可执行
✅ 相关问题已记录
```

---

## 🧪 测试

### 适用场景
- 编写新测试
- 修复失败测试
- 提升测试覆盖率
- 集成测试

### 分步工作流

#### 第 1 步：明确测试内容
```yaml
测试类型：
  1. 单元测试
     - 单个函数/方法
     - 业务逻辑
     - 数据变换

  2. 集成测试
     - API 接口
     - 数据库操作
     - 服务交互

  3. 端到端测试
     - 完整用户工作流
     - 多组件交互
```

#### 第 2 步：编写测试
```yaml
结构（AAA 模式）：
  1. Arrange：准备测试数据与环境
  2. Act：执行被测代码
  3. Assert：断言预期结果

示例（Python/pytest）：
  def test_wholesale_price_calculation():
      # Arrange
      product = Product(price=100.00)
      client = Client(discount_percent=15)
      quantity = 150

      # Act
      final_price = calculate_wholesale_price(product, client, quantity)

      # Assert
      assert final_price == 72.25  # 100 - 15% - 10%（量级）

测试覆盖：
  - 正常路径（预期输入）
  - 边界值（边界场景）
  - 错误场景（非法输入）
  - 阿拉伯语文本（面向用户的功能）
```

#### 第 3 步：运行测试
```yaml
Actions：
  1. 先本地运行
     - pytest tests/
     - npm test
     - flutter test

  2. 修复任何失败
     - 理解失败原因
     - 修复代码或修复错误的测试

  3. 验证覆盖率
     - pytest --cov=app tests/
     - 关键路径覆盖率目标 >80%
```

#### 第 4 步：集成 CI
```yaml
Actions：
  1. 确保测试在 CI 中运行
     - GitHub Actions 应运行测试
     - 测试必须在合并前通过

  2. 保持测试快速
     - 单元测试 < 1 秒/个
     - 集成测试 < 5 秒/个
     - Mock 外部服务（Zoho API）
```

### 质量门（必须通过）
```yaml
✅ 测试遵循 AAA 模式
✅ 测试清晰聚焦
✅ 覆盖正常路径 + 边界 + 错误
✅ 测试运行快速
✅ 本地全部通过
✅ CI 全部通过
```

---

## 🔐 安全增强

### 适用场景
- 添加安全特性
- 修复安全漏洞
- 实施认证/鉴权
- 进行安全审计

### 分步工作流

#### 第 1 步：评估现状
```yaml
Actions：
  1. 检查安全姿态
     - 是否启用认证？
     - 是否启用鉴权？
     - 输入校验？
     - 防 SQL 注入？
     - 防 XSS？
     - CSRF 保护？

  2. 识别漏洞
     - 使用安全扫描工具
     - 以常见问题审查代码
     - 检查依赖的 CVE
```

#### 第 2 步：设计安全增强
```yaml
Questions：
  1. 我们在应对何种威胁？
  2. 安全边界是什么？
  3. 对用户的影响？
  4. 性能影响？

安全分层：
  1. 认证（你是谁？）
  2. 鉴权（你能做什么？）
  3. 输入校验（是否安全？）
  4. 输出编码（无法注入）
  5. 审计日志（谁做了什么？）
```

#### 第 3 步：安全实施
```yaml
最佳实践：
  1. 使用成熟库
     - 不要自造加密
     - 使用 FastAPI 安全工具
     - 使用 SQLAlchemy（防 SQL 注入）

  2. 纵深防御
     - 多层安全措施
     - 安全失败（默认拒绝）

  3. 全面校验
     - 永远不信任用户输入
     - 后端校验（不只前端）
     - 使用 Pydantic schema

  4. 遵循 OWASP Top 10
     - 注入
     - 认证破坏
     - 敏感数据暴露
     - XML 外部实体（XXE）
     - 访问控制破坏
     - 安全错误配置
     - 跨站脚本（XSS）
     - 不安全反序列化
     - 使用已知漏洞组件
     - 日志与监控不足
```

#### 第 4 步：安全测试
```yaml
Actions：
  1. 攻击场景测试
     - 试图绕过认证
     - 尝试 SQL 注入
     - 尝试 XSS 攻击
     - 尝试未授权访问

  2. 安全扫描
     - 运行 bandit（Python 安全 linter）
     - 运行 npm audit（Node.js）
     - 检查依赖

  3. 渗透测试
     - 手工测试安全控制
     - 自动化安全扫描
```

#### 第 5 步：文档与监控
```yaml
Actions：
  1. 记录安全措施
     - 保护了什么
     - 如何保护
     - 假设条件

  2. 建立监控
     - 登录失败次数
     - 异常访问模式
     - 安全事件日志

  3. 事件响应预案
     - 发现入侵后如何处理
     - 联系谁
     - 如何隔离
```

### 质量门（必须通过）
```yaml
✅ 威胁清晰识别
✅ 遵循安全最佳实践
✅ 已实施输入校验
✅ 认证/鉴权已执行
✅ 防 SQL 注入（使用 ORM）
✅ 防 XSS（输出编码）
✅ 安全测试通过
✅ 无硬编码密钥
✅ 审计日志就位
```

---

## 📊 任务模式选择决策树

```
开始
  ↓
是否在添加新功能？
  是 → 🧩 功能实现
  否 → 继续
  ↓
是否有损坏的功能？
  是 → 🐛 Bug 修复
  否 → 继续
  ↓
是否在提升代码质量或性能？
  是 → 🧠 重构 / 优化
  否 → 继续
  ↓
是否在更新文档？
  是 → 🧾 文档更新
  否 → 继续
  ↓
是否在部署改动？
  是 → 🔄 部署 / CI/CD 任务
  否 → 继续
  ↓
是否在搜索/理解代码？
  是 → 🔍 调研 / 研究
  否 → 继续
  ↓
是否在编写/修复测试？
  是 → 🧪 测试
  否 → 继续
  ↓
是否与安全相关？
  是 → 🔐 安全增强
  否 → 询问 Khaleel 以澄清
```

---

## ✅ 通用质量原则

适用于所有任务类型：

```yaml
开始前：
  - [ ] 阅读 PROJECT_VISION.md（理解业务背景）
  - [ ] 阅读 ARCHITECTURE_RULES.md（遵循模式）
  - [ ] 搜索现有代码（优先增强而非新建）
  - [ ] 若不确定则提出澄清问题

进行中：
  - [ ] 遵循命名约定
  - [ ] 包含阿拉伯语支持（若面向用户）
  - [ ] 添加校验与错误处理
  - [ ] 编写清晰 docstrings 与注释
  - [ ] 持续测试

完成前：
  - [ ] 本地已测试
  - [ ] 代码整洁可读
  - [ ] 文档已更新（如需）
  - [ ] 使用清晰信息提交
  - [ ] 部署到预发布（若为部署任务）
  - [ ] 验证工作正常
```

---

## 🎯 成功指标

当你有效地使用任务模式时：

- ✅ 各任务类型均遵循一致步骤
- ✅ 不跳过关键步骤（特别是上下文加载）
- ✅ 质量门在标记完成前均已通过
- ✅ Khaleel 不需重复同样指令
- ✅ 功能首次即可正确工作
- ✅ 代码遵循既定模式
- ✅ 文档保持更新

---

**任务模式（中文版）结束**