#!/usr/bin/env bash
# ============================================================
# Script: {{SCRIPT_NAME}}.sh
# Purpose: {{DESCRIPTION}}
# Generated by: cli-ifying skill
# Source: {{SOURCE_TYPE}}:{{SOURCE_NAME}}
# ============================================================
set -euo pipefail

# Version
readonly VERSION="1.0.0"
readonly SCRIPT_NAME="{{SCRIPT_NAME}}"

# Colors (only if terminal supports it)
if [[ -t 1 ]] && [[ "${TERM:-}" != "dumb" ]]; then
    readonly RED='\033[0;31m'
    readonly GREEN='\033[0;32m'
    readonly YELLOW='\033[0;33m'
    readonly BLUE='\033[0;34m'
    readonly NC='\033[0m' # No Color
else
    readonly RED=''
    readonly GREEN=''
    readonly YELLOW=''
    readonly BLUE=''
    readonly NC=''
fi

# ============================================================
# Configuration (from environment with defaults)
# ============================================================
: "${OUTPUT_FORMAT:=json}"
: "${VERBOSE:=false}"
: "${QUIET:=false}"

# Required environment variables (uncomment and customize)
# : "${API_KEY:?API_KEY environment variable required}"
# : "${CONFIG_PATH:=${HOME}/.config/${SCRIPT_NAME}}"

# ============================================================
# Logging & Output Helpers
# ============================================================

log_error() {
    echo -e "${RED}ERROR:${NC} $*" >&2
}

log_warn() {
    echo -e "${YELLOW}WARN:${NC} $*" >&2
}

log_info() {
    [[ "$QUIET" == "true" ]] && return
    echo -e "${BLUE}INFO:${NC} $*" >&2
}

log_debug() {
    [[ "$VERBOSE" != "true" ]] && return
    echo -e "DEBUG: $*" >&2
}

# Output JSON (for programmatic consumption)
output_json() {
    echo "$1"
}

# Output formatted text (for human reading)
output_text() {
    if command -v jq &>/dev/null; then
        echo "$1" | jq -r '.'
    else
        echo "$1"
    fi
}

# Format output based on OUTPUT_FORMAT
format_output() {
    local data="$1"
    case "$OUTPUT_FORMAT" in
        json) output_json "$data" ;;
        text) output_text "$data" ;;
        *)    output_json "$data" ;;
    esac
}

# ============================================================
# Help & Version
# ============================================================

show_version() {
    echo "$SCRIPT_NAME version $VERSION"
}

show_help() {
    cat << 'EOF'
{{SCRIPT_NAME}} - {{DESCRIPTION}}

USAGE:
    {{SCRIPT_NAME}} <command> [options] [arguments]

COMMANDS:
    run                 Execute the main operation
    list                List resources
    get <id>            Get a specific resource
    help                Show this help message
    version             Show version information

OPTIONS:
    -o, --output FORMAT   Output format: json (default) | text
    -q, --quiet           Suppress non-essential output
    -v, --verbose         Enable verbose/debug output
    -h, --help            Show this help message
    --version             Show version information

ENVIRONMENT VARIABLES:
    OUTPUT_FORMAT         Default output format (json)
    VERBOSE               Enable verbose mode (true/false)
    QUIET                 Enable quiet mode (true/false)

EXAMPLES:
    # Run the main operation
    {{SCRIPT_NAME}} run

    # List resources as JSON
    {{SCRIPT_NAME}} list

    # Get specific resource
    {{SCRIPT_NAME}} get abc123

    # Pipe output to jq for processing
    {{SCRIPT_NAME}} list | jq '.[0].id'

    # Text output for humans
    {{SCRIPT_NAME}} list --output text

EXIT CODES:
    0   Success
    1   General error
    2   Invalid arguments / usage error
    3   Missing dependencies
    4   Network/API error
    5   Permission denied

NOTES:
    Generated from {{SOURCE_TYPE}}:{{SOURCE_NAME}}
    See: ~/.claude/skills/cli-ifying/
EOF
}

# ============================================================
# Dependency Checks
# ============================================================

check_dependencies() {
    local missing=()

    # Add required commands here
    local deps=(
        "jq"
        "curl"
        # "git"
        # "gh"
    )

    for cmd in "${deps[@]}"; do
        if ! command -v "$cmd" &>/dev/null; then
            missing+=("$cmd")
        fi
    done

    if [[ ${#missing[@]} -gt 0 ]]; then
        log_error "Missing required dependencies: ${missing[*]}"
        log_error "Please install them and try again."
        exit 3
    fi
}

# ============================================================
# Command Implementations
# ============================================================

# Main operation - customize this
cmd_run() {
    log_info "Running main operation..."

    # {{TODO: Implement main operation}}
    # Example:
    # local result
    # result=$(some_operation)
    # format_output "$result"

    echo '{"status": "success", "message": "Operation completed"}'
}

# List resources - customize this
cmd_list() {
    log_info "Listing resources..."

    # {{TODO: Implement list operation}}
    # Example:
    # local result
    # result=$(curl -s "https://api.example.com/items" -H "Authorization: Bearer $API_KEY")
    # format_output "$result"

    echo '[]'
}

# Get specific resource - customize this
cmd_get() {
    local id="${1:-}"

    if [[ -z "$id" ]]; then
        log_error "ID argument required"
        echo "Usage: $SCRIPT_NAME get <id>"
        exit 2
    fi

    log_info "Getting resource: $id"

    # {{TODO: Implement get operation}}
    # Example:
    # local result
    # result=$(curl -s "https://api.example.com/items/$id" -H "Authorization: Bearer $API_KEY")
    # format_output "$result"

    echo "{\"id\": \"$id\", \"found\": false}"
}

# ============================================================
# Argument Parsing
# ============================================================

parse_args() {
    local positional=()

    while [[ $# -gt 0 ]]; do
        case "$1" in
            -o|--output)
                OUTPUT_FORMAT="${2:-json}"
                shift 2
                ;;
            -q|--quiet)
                QUIET=true
                shift
                ;;
            -v|--verbose)
                VERBOSE=true
                shift
                ;;
            -h|--help)
                show_help
                exit 0
                ;;
            --version)
                show_version
                exit 0
                ;;
            --)
                shift
                positional+=("$@")
                break
                ;;
            -*)
                log_error "Unknown option: $1"
                show_help >&2
                exit 2
                ;;
            *)
                positional+=("$1")
                shift
                ;;
        esac
    done

    # Set positional arguments
    set -- "${positional[@]}"

    # Command is first positional argument
    COMMAND="${1:-help}"
    shift || true
    ARGS=("$@")
}

# ============================================================
# Main Entry Point
# ============================================================

main() {
    parse_args "$@"

    # Check dependencies before running commands
    case "$COMMAND" in
        help|--help|-h|version|--version)
            # No dependency check needed for help/version
            ;;
        *)
            check_dependencies
            ;;
    esac

    # Dispatch to command handler
    case "$COMMAND" in
        run)
            cmd_run "${ARGS[@]}"
            ;;
        list)
            cmd_list "${ARGS[@]}"
            ;;
        get)
            cmd_get "${ARGS[@]}"
            ;;
        help)
            show_help
            ;;
        version)
            show_version
            ;;
        *)
            log_error "Unknown command: $COMMAND"
            echo "Run '$SCRIPT_NAME help' for usage information." >&2
            exit 2
            ;;
    esac
}

# Run main only if script is executed (not sourced)
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi
