---
name: fullstory-observe-callbacks-mobile
version: v3
platforms:
  - ios
  - android
  - flutter
  - react-native
sdks:
  - fullstory-swift
  - fullstory-android
  - fullstory-flutter
  - fullstory-react-native
description: Mobile platform guide for Fullstory lifecycle callbacks. Explains platform-specific patterns since the observe() pattern is web-only.
parent_skill: fullstory-observe-callbacks
related_skills:
  - fullstory-observe-callbacks
  - fullstory-async-methods
  - fullstory-capture-control
---

# Fullstory Observe (Callbacks) — Mobile Platforms

> **Core Concepts**: See [SKILL.md](./SKILL.md) for general callback concepts.
>
> **Web Implementation**: See [SKILL-WEB.md](./SKILL-WEB.md) for the JavaScript `FS('observe', {...})` pattern.

---

## Platform Differences

The `FS('observe', {...})` pattern is **specific to the web/browser SDK**. Mobile platforms use different callback mechanisms native to each platform.

| Platform | Callback Mechanism |
|----------|-------------------|
| Web | `FS('observe', { type, callback })` |
| iOS | `FSDelegate` protocol |
| Android | `FS.SessionListener` interface |
| Flutter | Streams and callbacks |
| React Native | Event subscriptions |

---

## iOS (Swift)

iOS uses the `FSDelegate` protocol for lifecycle callbacks.

### FSDelegate Protocol

```swift
import FullStory

class AppDelegate: UIResponder, UIApplicationDelegate, FSDelegate {
    
    func application(_ application: UIApplication,
                     didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {
        
        // Set delegate before starting
        FS.delegate = self
        FS.start()
        
        return true
    }
    
    // MARK: - FSDelegate
    
    // Called when session URL becomes available (equivalent to 'session' observer)
    func fullstoryDidStartSession(_ sessionUrl: String) {
        print("Session started: \(sessionUrl)")
        
        // Forward to error tracking
        Sentry.configureScope { scope in
            scope.setTag(value: sessionUrl, key: "fullstory_url")
        }
        
        // Forward to support tool
        Intercom.setUserAttribute("fullstory_url", value: sessionUrl)
    }
    
    // Called when session ends
    func fullstoryDidStopSession() {
        print("Session stopped")
    }
}
```

### SwiftUI Integration

```swift
import SwiftUI
import FullStory

class FullstoryManager: ObservableObject, FSDelegate {
    @Published var sessionUrl: String?
    @Published var isCapturing: Bool = false
    
    init() {
        FS.delegate = self
    }
    
    func fullstoryDidStartSession(_ sessionUrl: String) {
        DispatchQueue.main.async {
            self.sessionUrl = sessionUrl
            self.isCapturing = true
        }
    }
    
    func fullstoryDidStopSession() {
        DispatchQueue.main.async {
            self.isCapturing = false
        }
    }
}

// Usage in SwiftUI
struct ContentView: View {
    @StateObject private var fullstory = FullstoryManager()
    
    var body: some View {
        VStack {
            if fullstory.isCapturing {
                Text("Session recording")
                if let url = fullstory.sessionUrl {
                    Button("Copy Session URL") {
                        UIPasteboard.general.string = url
                    }
                }
            }
        }
    }
}
```

---

## Android (Kotlin)

Android uses the `FS.SessionListener` interface for lifecycle callbacks.

### SessionListener Interface

```kotlin
import com.fullstory.FS

class MyApplication : Application(), FS.SessionListener {
    
    override fun onCreate() {
        super.onCreate()
        
        // Set listener before any FS calls
        FS.setSessionListener(this)
    }
    
    // Called when session URL becomes available (equivalent to 'session' observer)
    override fun onSessionChanged(sessionUrl: String) {
        Log.d("Fullstory", "Session started: $sessionUrl")
        
        // Forward to error tracking
        Sentry.configureScope { scope ->
            scope.setTag("fullstory_url", sessionUrl)
        }
        
        // Forward to support tool
        Intercom.client().setUserAttribute("fullstory_url", sessionUrl)
    }
}
```

### ViewModel Integration

```kotlin
import androidx.lifecycle.ViewModel
import androidx.lifecycle.MutableLiveData
import com.fullstory.FS

class FullstoryViewModel : ViewModel(), FS.SessionListener {
    
    val sessionUrl = MutableLiveData<String?>()
    val isCapturing = MutableLiveData<Boolean>(false)
    
    init {
        FS.setSessionListener(this)
    }
    
    override fun onSessionChanged(url: String) {
        sessionUrl.postValue(url)
        isCapturing.postValue(true)
    }
    
    override fun onCleared() {
        super.onCleared()
        // Note: Android SDK doesn't have a removeListener method
        // Listener is weak-referenced and cleaned up automatically
    }
}

// Usage in Fragment/Activity
class MyFragment : Fragment() {
    private val viewModel: FullstoryViewModel by viewModels()
    
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        
        viewModel.sessionUrl.observe(viewLifecycleOwner) { url ->
            url?.let {
                binding.sessionUrlText.text = it
            }
        }
    }
}
```

---

## Flutter (Dart)

Flutter uses streams and callbacks for lifecycle events.

### Stream-based Approach

```dart
import 'package:fullstory_flutter/fs.dart';
import 'dart:async';

class FullstoryManager {
  String? sessionUrl;
  bool isCapturing = false;
  StreamSubscription? _subscription;
  
  final _sessionController = StreamController<String>.broadcast();
  Stream<String> get sessionStream => _sessionController.stream;
  
  void initialize() {
    // Listen for session URL changes
    _subscription = FS.sessionUrlStream.listen((url) {
      sessionUrl = url;
      isCapturing = true;
      _sessionController.add(url);
      
      // Forward to error tracking
      Sentry.configureScope((scope) {
        scope.setTag('fullstory_url', url);
      });
    });
  }
  
  void dispose() {
    _subscription?.cancel();
    _sessionController.close();
  }
}

// Usage with Provider
class MyApp extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return Provider(
      create: (_) => FullstoryManager()..initialize(),
      dispose: (_, manager) => manager.dispose(),
      child: MaterialApp(
        home: HomeScreen(),
      ),
    );
  }
}

// In widget
class SupportButton extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    final fsManager = Provider.of<FullstoryManager>(context);
    
    return StreamBuilder<String>(
      stream: fsManager.sessionStream,
      builder: (context, snapshot) {
        return ElevatedButton(
          onPressed: () {
            openSupport(sessionUrl: snapshot.data);
          },
          child: Text(
            snapshot.hasData 
              ? 'Contact Support (Session attached)' 
              : 'Contact Support'
          ),
        );
      },
    );
  }
}
```

---

## React Native

React Native uses event subscriptions similar to web JavaScript.

### Event Subscription Pattern

```javascript
import FullStory from '@fullstory/react-native';
import { useEffect, useState } from 'react';

// Custom hook for session tracking
function useFullstorySession() {
  const [sessionUrl, setSessionUrl] = useState(null);
  const [isCapturing, setIsCapturing] = useState(false);
  
  useEffect(() => {
    // Subscribe to session events
    const subscription = FullStory.onSession((url) => {
      setSessionUrl(url);
      setIsCapturing(true);
      
      // Forward to error tracking
      Sentry.setTag('fullstory_url', url);
    });
    
    // Get current session URL if already available
    FullStory.getCurrentSessionURL().then((url) => {
      if (url) {
        setSessionUrl(url);
        setIsCapturing(true);
      }
    });
    
    // Cleanup subscription on unmount
    return () => {
      subscription.remove();
    };
  }, []);
  
  return { sessionUrl, isCapturing };
}

// Usage in component
function SupportButton() {
  const { sessionUrl, isCapturing } = useFullstorySession();
  
  const handlePress = () => {
    openSupport({
      fullstoryUrl: sessionUrl,
    });
  };
  
  return (
    <TouchableOpacity onPress={handlePress}>
      <Text>
        Contact Support
        {sessionUrl && ' (Session attached)'}
      </Text>
    </TouchableOpacity>
  );
}
```

### Class Component Pattern

```javascript
import React, { Component } from 'react';
import FullStory from '@fullstory/react-native';

class FullstoryProvider extends Component {
  state = {
    sessionUrl: null,
    isCapturing: false,
  };
  
  subscription = null;
  
  componentDidMount() {
    this.subscription = FullStory.onSession((url) => {
      this.setState({
        sessionUrl: url,
        isCapturing: true,
      });
    });
    
    // Check for existing session
    FullStory.getCurrentSessionURL().then((url) => {
      if (url) {
        this.setState({
          sessionUrl: url,
          isCapturing: true,
        });
      }
    });
  }
  
  componentWillUnmount() {
    if (this.subscription) {
      this.subscription.remove();
    }
  }
  
  render() {
    return (
      <FullstoryContext.Provider value={this.state}>
        {this.props.children}
      </FullstoryContext.Provider>
    );
  }
}
```

---

## Equivalent Patterns

### Session URL Callback

| Platform | Mechanism |
|----------|-----------|
| Web | `FS('observe', { type: 'session', callback })` |
| iOS | `FSDelegate.fullstoryDidStartSession(_:)` |
| Android | `FS.SessionListener.onSessionChanged()` |
| Flutter | `FS.sessionUrlStream.listen()` |
| React Native | `FullStory.onSession()` |

### Cleanup/Unsubscribe

| Platform | Pattern |
|----------|---------|
| Web | `observer.disconnect()` |
| iOS | Set `FS.delegate = nil` |
| Android | Weak reference, automatic cleanup |
| Flutter | `subscription.cancel()` |
| React Native | `subscription.remove()` |

---

## Best Practices (All Platforms)

1. **Set up callbacks early** — Configure before starting Fullstory
2. **Clean up on component unmount** — Prevent memory leaks
3. **Handle null/missing URLs** — Session may not be immediately available
4. **Forward to integrations** — Send URL to error tracking, support tools
5. **Update UI reactively** — Use platform-appropriate reactive patterns

---

## Reference Links

- **iOS SDK**: https://developer.fullstory.com/mobile/ios/
- **Android SDK**: https://developer.fullstory.com/mobile/android/
- **Flutter SDK**: https://developer.fullstory.com/mobile/flutter/
- **React Native SDK**: https://developer.fullstory.com/mobile/react-native/
