---
name: fullstory-observe-callbacks-web
version: v3
platform: web
sdk: fullstory-browser
description: JavaScript/TypeScript implementation guide for Fullstory's Observer/Callback API. Includes API reference, lifecycle management, and framework integration patterns.
parent_skill: fullstory-observe-callbacks
related_skills:
  - fullstory-observe-callbacks
  - fullstory-async-methods
  - fullstory-capture-control
---

# Fullstory Observe (Callbacks) — Web Implementation

> **Core Concepts**: See [SKILL.md](./SKILL.md) for observer types and best practices.
>
> **Other Platforms**: See [SKILL-MOBILE.md](./SKILL-MOBILE.md) for mobile platform differences.

---

## API Reference

### Basic Syntax

```javascript
const observer = FS('observe', {
  type: string,         // Required: 'start' or 'session'
  callback: function    // Required: Function called when event fires
});

// Later: stop observing
observer.disconnect();
```

### Async Version

```javascript
const observer = await FS('observeAsync', {
  type: string,
  callback: function
});

observer.disconnect();
```

### Parameters

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `type` | string | **Yes** | Event type: `'start'` or `'session'` |
| `callback` | function | **Yes** | Function called when event occurs |

### Callback Arguments

| Event Type | Callback Argument |
|------------|-------------------|
| `'start'` | `undefined` |
| `'session'` | `{ url: string }` |

---

## ✅ GOOD Implementation Examples

### Example 1: Capture Session URL for Third-party Tools

```javascript
// GOOD: Forward session URL to support/analytics tools
function initializeSessionTracking() {
  const observer = FS('observe', {
    type: 'session',
    callback: (session) => {
      const sessionUrl = session.url;
      
      // Send to support tool (e.g., Intercom, Zendesk)
      if (window.Intercom) {
        window.Intercom('update', {
          fullstory_url: sessionUrl
        });
      }
      
      // Send to error tracking (e.g., Sentry, Bugsnag)
      if (window.Sentry) {
        window.Sentry.setTag('fullstory_url', sessionUrl);
      }
      
      // Store for later use
      window.__fullstorySessionUrl = sessionUrl;
      
      console.log('Session URL captured:', sessionUrl);
    }
  });
  
  // Return cleanup function
  return () => observer.disconnect();
}

// Initialize on app load
const cleanup = initializeSessionTracking();

// On app cleanup (e.g., SPA unmount)
// cleanup();
```

**Why this is good:**
- ✅ Integrates with third-party tools
- ✅ Stores URL for later access
- ✅ Returns cleanup function
- ✅ Logs for debugging

### Example 2: React Hook for Session URL

```jsx
// GOOD: React hook for Fullstory session management
import { useState, useEffect, useCallback } from 'react';

function useFullstorySession() {
  const [sessionUrl, setSessionUrl] = useState(null);
  const [isCapturing, setIsCapturing] = useState(false);
  const [isReady, setIsReady] = useState(false);
  
  useEffect(() => {
    // Check if FS is available
    if (typeof FS === 'undefined') {
      return;
    }
    
    const observers = [];
    
    // Listen for capture start
    const startObserver = FS('observe', {
      type: 'start',
      callback: () => {
        setIsCapturing(true);
        setIsReady(true);
      }
    });
    observers.push(startObserver);
    
    // Listen for session URL
    const sessionObserver = FS('observe', {
      type: 'session',
      callback: (session) => {
        setSessionUrl(session.url);
      }
    });
    observers.push(sessionObserver);
    
    // Cleanup on unmount
    return () => {
      observers.forEach(obs => obs.disconnect());
    };
  }, []);
  
  const copySessionUrl = useCallback(() => {
    if (sessionUrl) {
      navigator.clipboard.writeText(sessionUrl);
      return true;
    }
    return false;
  }, [sessionUrl]);
  
  return {
    sessionUrl,
    isCapturing,
    isReady,
    copySessionUrl
  };
}

// Usage in component
function SupportButton() {
  const { sessionUrl, isReady } = useFullstorySession();
  
  const handleSupportClick = () => {
    openSupportChat({
      fullstoryUrl: sessionUrl
    });
  };
  
  return (
    <button 
      onClick={handleSupportClick}
      disabled={!isReady}
    >
      Contact Support
      {sessionUrl && ' (Session will be attached)'}
    </button>
  );
}
```

**Why this is good:**
- ✅ Proper React lifecycle handling
- ✅ Cleanup on unmount prevents memory leaks
- ✅ Exposes useful state (isCapturing, isReady)
- ✅ Helper function for copying URL
- ✅ Graceful handling if FS unavailable

### Example 3: Session URL for Error Reports

```javascript
// GOOD: Attach session URL to all error reports
class ErrorReporter {
  constructor() {
    this.sessionUrl = null;
    this.observer = null;
  }
  
  initialize() {
    if (typeof FS === 'undefined') {
      console.warn('Fullstory not available for error reporting');
      return;
    }
    
    // Set up session observer
    this.observer = FS('observe', {
      type: 'session',
      callback: (session) => {
        this.sessionUrl = session.url;
        
        // Update Sentry context
        if (window.Sentry) {
          window.Sentry.setContext('fullstory', {
            sessionUrl: session.url
          });
        }
        
        // Update Bugsnag
        if (window.Bugsnag) {
          window.Bugsnag.addMetadata('fullstory', {
            sessionUrl: session.url
          });
        }
      }
    });
    
    // Set up global error handler
    window.addEventListener('error', (event) => {
      this.reportError(event.error, {
        source: 'window.onerror',
        filename: event.filename,
        lineno: event.lineno
      });
    });
  }
  
  reportError(error, context = {}) {
    const report = {
      error: error?.message || String(error),
      stack: error?.stack,
      fullstoryUrl: this.sessionUrl,
      timestamp: new Date().toISOString(),
      ...context
    };
    
    // Send to your error service
    this.sendErrorReport(report);
    
    // Also log to Fullstory
    if (typeof FS !== 'undefined') {
      FS('log', {
        level: 'error',
        msg: report.error
      });
    }
  }
  
  sendErrorReport(report) {
    fetch('/api/errors', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(report)
    }).catch(console.error);
  }
  
  cleanup() {
    if (this.observer) {
      this.observer.disconnect();
    }
  }
}

// Initialize on app start
const errorReporter = new ErrorReporter();
errorReporter.initialize();
```

**Why this is good:**
- ✅ Integrates with popular error tools
- ✅ Auto-attaches session URL to all errors
- ✅ Also logs to Fullstory
- ✅ Cleanup method available

### Example 4: Async Observer with Error Handling

```javascript
// GOOD: Using async observers with error handling
async function setupFullstoryIntegration() {
  try {
    // Set up start observer
    const startObserver = await FS('observeAsync', {
      type: 'start',
      callback: () => {
        console.log('Fullstory started capturing');
        enableSessionReplayFeatures();
      }
    });
    
    // Set up session observer
    const sessionObserver = await FS('observeAsync', {
      type: 'session',
      callback: (session) => {
        console.log('Session URL:', session.url);
        notifyIntegrations(session.url);
      }
    });
    
    // Return combined cleanup
    return {
      cleanup: () => {
        startObserver.disconnect();
        sessionObserver.disconnect();
      },
      status: 'success'
    };
    
  } catch (error) {
    console.warn('Fullstory integration failed:', error);
    return {
      cleanup: () => {},
      status: 'failed',
      error
    };
  }
}

// Usage
let fsIntegration = null;

async function initApp() {
  fsIntegration = await setupFullstoryIntegration();
  
  if (fsIntegration.status === 'success') {
    showSessionReplayBadge();
  }
}

function cleanupApp() {
  if (fsIntegration) {
    fsIntegration.cleanup();
  }
}
```

**Why this is good:**
- ✅ Uses async version for proper error handling
- ✅ Returns status for conditional UI
- ✅ Combined cleanup function
- ✅ Graceful handling of failures

---

## ❌ BAD Implementation Examples

### Example 1: Not Disconnecting Observers

```javascript
// BAD: Observer never cleaned up
function setupSessionCallback() {
  FS('observe', {
    type: 'session',
    callback: (session) => {
      console.log('Session:', session.url);
    }
  });
  // Observer never stored or cleaned up!
}

// Called multiple times in SPA
setupSessionCallback();  // Leak
setupSessionCallback();  // Another leak
setupSessionCallback();  // More leaks...
```

**Why this is bad:**
- ❌ Observer never disconnected
- ❌ Memory leak in long-running apps
- ❌ Multiple callbacks pile up

**CORRECTED:**

```javascript
// GOOD: Store and cleanup observer
let sessionObserver = null;

function setupSessionCallback() {
  // Clean up existing observer first
  if (sessionObserver) {
    sessionObserver.disconnect();
  }
  
  sessionObserver = FS('observe', {
    type: 'session',
    callback: (session) => {
      console.log('Session:', session.url);
    }
  });
}

function cleanup() {
  if (sessionObserver) {
    sessionObserver.disconnect();
    sessionObserver = null;
  }
}
```

### Example 2: Wrong Event Type

```javascript
// BAD: Invalid event type
FS('observe', {
  type: 'ready',  // BAD: Not a valid type!
  callback: () => console.log('Ready!')
});

FS('observe', {
  type: 'url',  // BAD: Not a valid type!
  callback: (url) => console.log(url)
});
```

**Why this is bad:**
- ❌ 'ready' and 'url' are not valid types
- ❌ Callback will never fire
- ❌ No error thrown, silent failure

**CORRECTED:**

```javascript
// GOOD: Use valid event types
FS('observe', {
  type: 'start',  // Valid: when capture starts
  callback: () => console.log('Fullstory started!')
});

FS('observe', {
  type: 'session',  // Valid: when session URL ready
  callback: (session) => console.log('URL:', session.url)
});
```

### Example 3: Expecting URL from 'start' Callback

```javascript
// BAD: Expecting session data from 'start' callback
FS('observe', {
  type: 'start',
  callback: (session) => {
    // BAD: session is undefined for 'start' type!
    console.log('Session URL:', session.url);  // Error!
  }
});
```

**Why this is bad:**
- ❌ 'start' callback receives no arguments
- ❌ Will throw error accessing .url of undefined

**CORRECTED:**

```javascript
// GOOD: Use 'session' type for URL
FS('observe', {
  type: 'session',
  callback: (session) => {
    console.log('Session URL:', session.url);  // Works!
  }
});
```

### Example 4: Blocking on Observer

```javascript
// BAD: Blocking app initialization on observer
async function initApp() {
  let sessionUrl = null;
  
  // This may never resolve if FS is blocked!
  await new Promise((resolve) => {
    FS('observe', {
      type: 'session',
      callback: (session) => {
        sessionUrl = session.url;
        resolve();
      }
    });
  });
  
  // If Fullstory is blocked, this never runs
  startApp(sessionUrl);
}
```

**Why this is bad:**
- ❌ If FS blocked, promise never resolves
- ❌ App initialization hangs forever

**CORRECTED:**

```javascript
// GOOD: Non-blocking with timeout
async function initApp() {
  // Start app immediately
  startApp(null);
  
  // Set up observer but don't block
  if (typeof FS !== 'undefined') {
    FS('observe', {
      type: 'session',
      callback: (session) => {
        updateAppWithSessionUrl(session.url);
      }
    });
  }
}
```

---

## Common Patterns

### Pattern 1: Observer Manager Class

```javascript
// Centralized observer management
class FullstoryObserverManager {
  constructor() {
    this.observers = new Map();
  }
  
  register(name, type, callback) {
    // Disconnect existing observer with same name
    if (this.observers.has(name)) {
      this.observers.get(name).disconnect();
    }
    
    if (typeof FS === 'undefined') {
      console.warn(`Cannot register observer '${name}': FS not available`);
      return false;
    }
    
    const observer = FS('observe', { type, callback });
    this.observers.set(name, observer);
    return true;
  }
  
  unregister(name) {
    if (this.observers.has(name)) {
      this.observers.get(name).disconnect();
      this.observers.delete(name);
    }
  }
  
  unregisterAll() {
    this.observers.forEach(obs => obs.disconnect());
    this.observers.clear();
  }
}

// Usage
const fsObservers = new FullstoryObserverManager();

fsObservers.register('session', 'session', (session) => {
  console.log('Session URL:', session.url);
});

fsObservers.register('start', 'start', () => {
  console.log('Fullstory started');
});

// Cleanup
fsObservers.unregisterAll();
```

### Pattern 2: Vue Composable

```javascript
// Vue 3 composable for Fullstory session
import { ref, onMounted, onUnmounted } from 'vue';

export function useFullstorySession() {
  const sessionUrl = ref(null);
  const isCapturing = ref(false);
  const observers = [];
  
  onMounted(() => {
    if (typeof FS === 'undefined') return;
    
    observers.push(
      FS('observe', {
        type: 'start',
        callback: () => {
          isCapturing.value = true;
        }
      })
    );
    
    observers.push(
      FS('observe', {
        type: 'session',
        callback: (session) => {
          sessionUrl.value = session.url;
        }
      })
    );
  });
  
  onUnmounted(() => {
    observers.forEach(obs => obs.disconnect());
  });
  
  return {
    sessionUrl,
    isCapturing
  };
}
```

### Pattern 3: Angular Service

```typescript
import { Injectable, OnDestroy } from '@angular/core';
import { BehaviorSubject } from 'rxjs';

@Injectable({ providedIn: 'root' })
export class FullstoryService implements OnDestroy {
  private sessionUrl$ = new BehaviorSubject<string | null>(null);
  private observers: any[] = [];
  
  constructor() {
    this.initialize();
  }
  
  private initialize(): void {
    if (typeof FS === 'undefined') return;
    
    this.observers.push(
      FS('observe', {
        type: 'session',
        callback: (session: { url: string }) => {
          this.sessionUrl$.next(session.url);
        }
      })
    );
  }
  
  getSessionUrl$() {
    return this.sessionUrl$.asObservable();
  }
  
  ngOnDestroy(): void {
    this.observers.forEach(obs => obs.disconnect());
  }
}
```

### Pattern 4: Event Emitter Pattern

```javascript
// Event emitter for Fullstory events
class FullstoryEventEmitter {
  constructor() {
    this.listeners = { start: [], session: [] };
    this.state = { started: false, sessionUrl: null };
    this.observers = [];
    this.initialize();
  }
  
  initialize() {
    if (typeof FS === 'undefined') return;
    
    this.observers.push(
      FS('observe', {
        type: 'start',
        callback: () => {
          this.state.started = true;
          this.emit('start');
        }
      })
    );
    
    this.observers.push(
      FS('observe', {
        type: 'session',
        callback: (session) => {
          this.state.sessionUrl = session.url;
          this.emit('session', session);
        }
      })
    );
  }
  
  on(event, callback) {
    this.listeners[event]?.push(callback);
    
    // Fire immediately if event already occurred
    if (event === 'start' && this.state.started) {
      callback();
    }
    if (event === 'session' && this.state.sessionUrl) {
      callback({ url: this.state.sessionUrl });
    }
    
    return () => this.off(event, callback);
  }
  
  off(event, callback) {
    const idx = this.listeners[event]?.indexOf(callback);
    if (idx > -1) {
      this.listeners[event].splice(idx, 1);
    }
  }
  
  emit(event, data) {
    this.listeners[event]?.forEach(cb => cb(data));
  }
  
  destroy() {
    this.observers.forEach(obs => obs.disconnect());
    this.listeners = { start: [], session: [] };
  }
}

// Global instance
const fsEvents = new FullstoryEventEmitter();

// Usage
const unsubscribe = fsEvents.on('session', (session) => {
  console.log('Session URL:', session.url);
});

// Later
unsubscribe();
```

---

## Reference Links

- **Callbacks and Delegates**: https://developer.fullstory.com/browser/fullcapture/callbacks-and-delegates/
- **Get Session Details**: https://developer.fullstory.com/browser/get-session-details/
- **Asynchronous Methods**: https://developer.fullstory.com/browser/asynchronous-methods/
