---
name: fullstory-observe-callbacks
version: v3
description: Core concepts for Fullstory's Observer/Callback API. Covers event subscription, observer lifecycle, cleanup patterns, and reacting to Fullstory lifecycle events. Note - the observe() pattern is web-only; mobile platforms use delegates and listeners.
platforms:
  - web
related_skills:
  - fullstory-async-methods
  - fullstory-capture-control
  - fullstory-logging
implementation_files:
  - SKILL-WEB.md
  - SKILL-MOBILE.md
---

# Fullstory Observe (Callbacks & Delegates) API

> **Implementation Files**: This document covers core concepts. For code examples, see:
> - [SKILL-WEB.md](./SKILL-WEB.md) — JavaScript/TypeScript (Browser)
> - [SKILL-MOBILE.md](./SKILL-MOBILE.md) — Platform differences for iOS, Android, Flutter, React Native

> **Note**: The `FS('observe', {...})` pattern is specific to the **web/browser SDK**. Mobile SDKs use platform-native patterns (delegates, listeners, streams).

## Overview

Fullstory's Observer API allows developers to register callbacks that react to Fullstory lifecycle events. Instead of polling or guessing when Fullstory is ready, you can subscribe to specific events and be notified when they occur. This is essential for:

- **Session URL Capture**: Get notified when session URL is available
- **Initialization Handling**: Know when Fullstory starts capturing
- **Third-party Integration**: Forward session URLs to other tools
- **Conditional Features**: Enable features based on FS state
- **Resource Management**: Clean up observers on component unmount

---

## Core Concepts

### Observer Types

| Type | Fires When | Callback Receives |
|------|------------|-------------------|
| `'start'` | Fullstory begins capturing | `undefined` |
| `'session'` | Session URL becomes available | `{ url: string }` |

### Observer Lifecycle

```
FS('observe', {...})  →  Returns Observer  →  Callback fires when event occurs
                              ↓
                    Call observer.disconnect()  →  Stops listening
```

### Key Behaviors

| Behavior | Description |
|----------|-------------|
| **Immediate callback** | If event already occurred, callback fires immediately |
| **Multiple observers** | Can register multiple observers for same event |
| **Disconnect cleanup** | Must disconnect to prevent memory leaks |
| **Async version** | Use `observeAsync` to wait for registration |

---

## Event Types

### `'start'` Event

- Fires when Fullstory begins capturing the session
- Callback receives `undefined` (no arguments)
- Use for: Enabling session-dependent features, initialization logging

### `'session'` Event

- Fires when session URL becomes available
- Callback receives `{ url: string }` object
- Use for: Forwarding URL to support tools, error tracking, logging

---

## Best Practices

### 1. Always Store and Disconnect Observers

```javascript
// Store reference
const observer = FS('observe', { type: 'session', callback: fn });

// Later: cleanup
observer.disconnect();
```

### 2. Handle Already-Occurred Events

Observers fire immediately if the event has already occurred. Design callbacks to handle both immediate and delayed invocation.

### 3. Use Framework Lifecycle Methods

- React: `useEffect` with cleanup return
- Vue: `onMounted` / `onUnmounted`
- Angular: `ngOnInit` / `ngOnDestroy`

### 4. Don't Block on Observer Callbacks

Observer registration is synchronous, but callbacks are asynchronous. Don't wrap in Promise expecting callback to resolve it reliably.

---

## Troubleshooting

### Callback Never Fires

| Cause | Solution |
|-------|----------|
| Fullstory not loaded | Verify FS is defined |
| Wrong event type | Use 'start' or 'session' only |
| FS not capturing | Check Fullstory console |

### Multiple Callbacks

| Cause | Solution |
|-------|----------|
| Multiple observers | Store and reuse reference |
| No cleanup in SPA | Disconnect on unmount |

### Memory Leaks

| Cause | Solution |
|-------|----------|
| Observers not disconnected | Always call disconnect() |
| New observers on navigation | Clean up in lifecycle methods |

---

## Key Takeaways for Agent

When helping developers with Observer API:

1. **Always emphasize**:
   - Store observer reference
   - Call disconnect() on cleanup
   - Use 'session' for URL, 'start' for capture start
   - Handle FS not being available

2. **Common mistakes to watch for**:
   - Not disconnecting observers (memory leak)
   - Wrong event type
   - Expecting URL from 'start' callback
   - Blocking on observer callback
   - Multiple observers without cleanup

3. **Questions to ask developers**:
   - Is this a SPA or traditional app?
   - Do you need the session URL or just capture status?
   - What framework are you using?
   - How will you clean up the observer?

4. **Platform routing**:
   - Web (JavaScript/TypeScript) → See SKILL-WEB.md
   - Mobile platforms → See SKILL-MOBILE.md for delegates/listeners

---

## Reference Links

- **Callbacks and Delegates**: https://developer.fullstory.com/browser/fullcapture/callbacks-and-delegates/
- **Get Session Details**: https://developer.fullstory.com/browser/get-session-details/
- **Asynchronous Methods**: https://developer.fullstory.com/browser/asynchronous-methods/
