# Common Optimization Scenarios

## Contents
- Slow pagination queries
- N+1 in application code
- Dashboard aggregation queries  
- Search/filter endpoints
- Bulk data operations
- Time-series queries

## Slow Pagination Queries

**Symptom:** Pages load slowly as user goes deeper

**Root cause:** `OFFSET n` scans and discards n rows

**Solution - Keyset pagination:**
```sql
-- Instead of
SELECT * FROM posts ORDER BY created_at DESC LIMIT 20 OFFSET 1000;

-- Use cursor-based
SELECT * FROM posts 
WHERE created_at < '2024-01-15 10:30:00'  -- last item from previous page
ORDER BY created_at DESC 
LIMIT 20;
```

**Required index:** `CREATE INDEX idx_posts_created_at ON posts(created_at DESC);`

## N+1 in Application Code

**Symptom:** Hundreds of queries for a single page load

**Detection pattern:**
```python
# Bad: N+1
users = db.query("SELECT * FROM users LIMIT 100")
for user in users:
    orders = db.query(f"SELECT * FROM orders WHERE user_id = {user.id}")
```

**Solution - JOIN or batch:**
```sql
-- Single query with JOIN
SELECT u.*, o.* 
FROM users u
LEFT JOIN orders o ON u.id = o.user_id;

-- Or batch with IN
SELECT * FROM orders WHERE user_id IN (1, 2, 3, ...);
```

**ORM hint:** Use `include`/`prefetch_related`/`eager_load` depending on framework.

## Dashboard Aggregation Queries

**Symptom:** Dashboard takes 10+ seconds to load

**Common issues:**
- Aggregating millions of rows in real-time
- Multiple sequential queries instead of single query
- Missing indexes on GROUP BY columns

**Solutions:**

1. **Pre-aggregate with materialized views:**
```sql
CREATE MATERIALIZED VIEW daily_sales AS
SELECT DATE(created_at) as day, SUM(amount) as total
FROM orders GROUP BY DATE(created_at);

REFRESH MATERIALIZED VIEW CONCURRENTLY daily_sales;
```

2. **Use summary tables updated by triggers/jobs**

3. **Add covering indexes:**
```sql
CREATE INDEX idx_orders_date_amount ON orders(created_at, amount);
```

## Search/Filter Endpoints

**Symptom:** Search API slow with multiple optional filters

**Common issues:**
- OR conditions across columns
- LIKE '%term%' patterns
- No index for filter combinations

**Solutions:**

1. **Full-text search for text:**
```sql
-- PostgreSQL
CREATE INDEX idx_products_search ON products USING gin(to_tsvector('english', name || ' ' || description));

SELECT * FROM products WHERE to_tsvector('english', name || ' ' || description) @@ to_tsquery('laptop');
```

2. **Partial indexes for common filters:**
```sql
CREATE INDEX idx_orders_pending ON orders(created_at) WHERE status = 'pending';
```

3. **Decompose OR into UNION ALL:**
```sql
SELECT * FROM products WHERE category_id = 5
UNION ALL
SELECT * FROM products WHERE brand_id = 10;
```

## Bulk Data Operations

**Symptom:** Import/export jobs timeout or lock tables

**Solutions:**

1. **Batch inserts:**
```sql
-- Instead of 10,000 individual INSERTs
INSERT INTO logs (msg, created_at) VALUES 
  ('msg1', now()), ('msg2', now()), ... -- 1000 at a time
```

2. **Disable indexes during bulk load:**
```sql
ALTER TABLE logs DISABLE TRIGGER ALL;
-- bulk insert
ALTER TABLE logs ENABLE TRIGGER ALL;
REINDEX TABLE logs;
```

3. **Use COPY (PostgreSQL) or LOAD DATA (MySQL):**
```sql
COPY logs FROM '/path/to/data.csv' WITH CSV HEADER;
```

## Time-Series Queries

**Symptom:** Queries on recent data as slow as queries on old data

**Solutions:**

1. **Partition by time:**
```sql
CREATE TABLE logs (
    id SERIAL,
    created_at TIMESTAMP,
    message TEXT
) PARTITION BY RANGE (created_at);

CREATE TABLE logs_2024_01 PARTITION OF logs
FOR VALUES FROM ('2024-01-01') TO ('2024-02-01');
```

2. **BRIN indexes for time columns:**
```sql
CREATE INDEX idx_logs_created_brin ON logs USING brin(created_at);
```

3. **Archive old data:**
```sql
-- Move to archive table
INSERT INTO logs_archive SELECT * FROM logs WHERE created_at < now() - interval '90 days';
DELETE FROM logs WHERE created_at < now() - interval '90 days';
```

## Quick Diagnosis Commands

### PostgreSQL
```sql
-- Find slow queries
SELECT query, calls, mean_time, total_time 
FROM pg_stat_statements ORDER BY mean_time DESC LIMIT 10;

-- Find missing indexes
SELECT relname, seq_scan, idx_scan, seq_tup_read
FROM pg_stat_user_tables 
WHERE seq_scan > idx_scan ORDER BY seq_tup_read DESC;

-- Table bloat
SELECT schemaname, relname, n_dead_tup, last_vacuum
FROM pg_stat_user_tables ORDER BY n_dead_tup DESC;
```

### MySQL
```sql
-- Find slow queries
SELECT * FROM sys.statements_with_runtimes_in_95th_percentile;

-- Unused indexes
SELECT * FROM sys.schema_unused_indexes;

-- Tables without primary key
SELECT * FROM sys.schema_tables_with_full_table_scans;
```

### MongoDB
```javascript
// Current operations
db.currentOp({ "secs_running": { "$gt": 5 } })

// Index usage stats
db.collection.aggregate([{ $indexStats: {} }])

// Collection stats
db.collection.stats()
```
