# NoSQL Optimization Patterns

## Quick Navigation
- [MongoDB](#mongodb) - ESR rule, compound indexes, aggregation pipeline
- [Redis](#redis) - data structures, key design, SCAN vs KEYS
- [DynamoDB](#dynamodb) - partition keys, GSI/LSI, query vs scan
- [Elasticsearch](#elasticsearch) - mappings, filter vs query context

## Contents
- MongoDB (index strategies, query optimization, aggregation pipeline, profiling)
- Redis (data structure selection, key design, memory optimization)
- DynamoDB (key design, access patterns, query optimization, cost optimization)
- Elasticsearch (mapping optimization, query optimization, performance tips)

## MongoDB

### Index Strategies
```javascript
// Compound index (order matters - ESR rule: Equality, Sort, Range)
db.orders.createIndex({ status: 1, created_at: -1, amount: 1 })

// Partial index (reduces size)
db.orders.createIndex(
  { customer_id: 1 },
  { partialFilterExpression: { status: "active" } }
)

// Text index for search
db.products.createIndex({ name: "text", description: "text" })
```

### Query Optimization
```javascript
// Use projection - only fetch needed fields
db.users.find({ active: true }, { name: 1, email: 1, _id: 0 })

// Covered query (all fields from index)
db.orders.find(
  { status: "pending" },
  { status: 1, created_at: 1, _id: 0 }
).hint({ status: 1, created_at: 1 })

// Avoid $where and JavaScript execution
// Bad
db.users.find({ $where: "this.age > 21" })
// Good
db.users.find({ age: { $gt: 21 } })
```

### Aggregation Pipeline
```javascript
// Filter early with $match
db.orders.aggregate([
  { $match: { status: "completed" } },  // First!
  { $group: { _id: "$customer_id", total: { $sum: "$amount" } } },
  { $sort: { total: -1 } },
  { $limit: 10 }
])

// Use $project to reduce document size between stages
// Avoid $lookup on large collections without indexes
```

### Anti-patterns
- Unbounded arrays (embed vs reference decision)
- Missing compound index order (ESR rule)
- Large `$in` arrays (>1000 elements)
- Regex without anchors: `/pattern/` vs `/^pattern/`

### Profiling
```javascript
db.setProfilingLevel(1, { slowms: 100 })
db.system.profile.find().sort({ ts: -1 }).limit(10)
db.collection.find({}).explain("executionStats")
```

## Redis

### Data Structure Selection
| Use Case | Structure | Example |
|----------|-----------|---------|
| Cache | STRING | `SET user:123 "{json}"` |
| Counters | STRING with INCR | `INCR page:views:home` |
| Leaderboards | SORTED SET | `ZADD leaderboard 100 "user:1"` |
| Unique items | SET | `SADD online:users "user:1"` |
| Time-series | SORTED SET / Stream | `ZADD events <timestamp> data` |
| Queue | LIST | `LPUSH / BRPOP` |
| Session | HASH | `HSET session:abc user_id 123` |

### Key Design
```
# Good: Hierarchical, scannable
user:123:profile
user:123:orders:2024
product:sku:ABC123

# Bad: Unpredictable patterns
data_user_123_profile
```

### Optimization Tips
- Use MGET/MSET for batch operations
- Pipeline commands to reduce round trips
- Set appropriate TTLs to control memory
- Use SCAN instead of KEYS in production
- Consider Redis Cluster for horizontal scaling

### Memory Optimization
- Use HASH for small objects (ziplist encoding)
- Compress large values client-side
- Use shorter key names in high-volume scenarios
- Monitor with `INFO memory` and `MEMORY USAGE key`

## DynamoDB

### Key Design (Single Table)
```
PK: USER#<user_id>  SK: PROFILE
PK: USER#<user_id>  SK: ORDER#<order_id>
PK: ORDER#<order_id> SK: ITEM#<item_id>
```

### Access Patterns
- Design for access patterns first
- Use GSIs for alternate query patterns
- LSIs for sorting within partition

### Query Optimization
```python
# Use ProjectionExpression
response = table.query(
    KeyConditionExpression=Key('pk').eq('USER#123'),
    ProjectionExpression='order_id, amount, #s',
    ExpressionAttributeNames={'#s': 'status'}
)

# Use FilterExpression for post-query filtering (still reads all)
# Prefer GSI if filtering removes >50% of items
```

### Anti-patterns
- Hot partitions (uneven key distribution)
- Scan operations (use Query with indexes)
- Large items (>400KB)
- Missing pagination (LastEvaluatedKey)

### Cost Optimization
- Use BatchGetItem/BatchWriteItem
- Enable auto-scaling or on-demand capacity
- Use sparse indexes (only items with attribute)
- Consider DAX for read-heavy workloads

## Elasticsearch

### Mapping Optimization
```json
{
  "mappings": {
    "properties": {
      "title": { "type": "text", "analyzer": "english" },
      "status": { "type": "keyword" },
      "created_at": { "type": "date" },
      "tags": { "type": "keyword" },
      "description": { "type": "text", "index": false }
    }
  }
}
```

### Query Optimization
- Use `filter` context for exact matches (cached, no scoring)
- Use `query` context only when relevance scoring needed
- Prefer `term` over `match` for keyword fields

```json
{
  "query": {
    "bool": {
      "filter": [
        { "term": { "status": "published" } },
        { "range": { "created_at": { "gte": "2024-01-01" } } }
      ],
      "must": [
        { "match": { "title": "search terms" } }
      ]
    }
  }
}
```

### Performance Tips
- Use `_source` filtering to reduce payload
- Avoid deep pagination (use `search_after`)
- Use index aliases for zero-downtime reindexing
- Consider routing for related documents
- Monitor with `_stats` and `_cat` APIs
