#!/usr/bin/env python3
"""
Detect ORM anti-patterns in application code.
Supports: SQLAlchemy, Django ORM, ActiveRecord patterns, Prisma, Sequelize

Usage: python detect_orm_issues.py app.py
       python detect_orm_issues.py --dir ./models --json
"""

import re
import sys
import json
import argparse
from pathlib import Path
from dataclasses import dataclass, asdict
from typing import List, Optional

@dataclass
class ORMIssue:
    file: str
    line: int
    severity: str
    pattern: str
    code_snippet: str
    problem: str
    solution: str

@dataclass
class AnalysisResult:
    files_analyzed: int
    issues: List[ORMIssue]
    summary: str

# Pattern definitions: (regex, severity, pattern_name, problem, solution)
PATTERNS = [
    # N+1 Query Patterns
    (
        r'for\s+\w+\s+in\s+.*?\.(?:all|objects|query|find).*?:.*?\.(?:filter|get|find|where|select)',
        'high',
        'N+1 Query (loop with query)',
        'Each iteration executes a separate query',
        'Use prefetch_related/select_related (Django), joinedload/selectinload (SQLAlchemy), or includes (ActiveRecord)'
    ),
    (
        r'\.forEach\s*\(\s*(?:async\s*)?\(?.*?\)?\s*=>\s*\{[^}]*(?:findOne|findUnique|findFirst)',
        'high',
        'N+1 Query (JS forEach with query)',
        'Each iteration executes a separate database query',
        'Use findMany with where IN clause, or use include for relations'
    ),
    
    # Missing eager loading
    (
        r'\.(?:all|objects\.all|find)\(\s*\)(?!.*(?:select_related|prefetch_related|includes?|joinedload|with))',
        'medium',
        'Potential lazy loading',
        'Accessing relations later may trigger additional queries',
        'Add eager loading if relations are accessed'
    ),
    
    # SELECT * equivalent
    (
        r'\.(?:get|all|find|first|last)\(\s*\)(?!.*(?:only|values|values_list|select|pluck|defer))',
        'low',
        'Fetching all columns',
        'Loading unnecessary columns increases memory and I/O',
        'Use only()/values()/select() to fetch needed columns'
    ),
    
    # Unbounded queries
    (
        r'\.(?:all|objects\.all|find)\(\s*\)(?!.*(?:limit|take|first|last|\[:|\[0:))',
        'medium',
        'Unbounded query',
        'Loading entire table into memory',
        'Add LIMIT/pagination or use iterator/cursor'
    ),
    
    # Count then fetch anti-pattern
    (
        r'(?:len\(.*?\.all\(\)|\.count\(\).*?\.all\(\)|\.length.*?\.find\()',
        'medium',
        'Count then fetch',
        'Two queries when one would suffice',
        'Use single query with count, or paginate with offset'
    ),
    
    # Raw SQL with string formatting (SQL injection risk + no parameterization)
    (
        r'(?:execute|raw|query)\s*\(\s*f["\']|(?:execute|raw|query)\s*\([^)]*%\s*\(|(?:execute|raw|query)\s*\([^)]*\.format\(',
        'high',
        'Unsafe raw SQL',
        'String interpolation in SQL prevents query plan caching and risks injection',
        'Use parameterized queries with placeholders'
    ),
    
    # Django: filter in loop
    (
        r'for\s+\w+\s+in\s+\w+:\s*\n\s*.*?\.objects\.filter\(',
        'high',
        'Django N+1 (filter in loop)',
        'Running filter() inside loop creates N+1 queries',
        'Use Q objects, __in lookup, or prefetch_related'
    ),
    
    # SQLAlchemy: query in loop
    (
        r'for\s+\w+\s+in\s+.*?:\s*\n\s*.*?session\.query\(',
        'high',
        'SQLAlchemy N+1 (query in loop)',
        'Running query() inside loop creates N+1 queries',
        'Use joinedload(), selectinload(), or single query with IN clause'
    ),
    
    # Prisma: findMany without include warning
    (
        r'prisma\.\w+\.findMany\(\s*\{(?!.*include)',
        'low',
        'Prisma findMany without include',
        'Related data may require additional queries',
        'Add include: {} if accessing relations'
    ),
    
    # Sequelize: findAll in loop
    (
        r'for\s*\(.*?\)\s*\{[^}]*\.findAll\(',
        'high',
        'Sequelize N+1 (findAll in loop)',
        'Running findAll inside loop creates N+1 queries',
        'Use include option or Op.in for batch queries'
    ),
    
    # save() in loop
    (
        r'for\s+.*?:\s*\n(?:[^\n]*\n)*?[^\n]*\.save\(\)',
        'high',
        'Save in loop',
        'Individual saves cause N database round-trips',
        'Use bulk_create/bulk_update (Django), add_all (SQLAlchemy), or insert_many'
    ),
    
    # create() in loop
    (
        r'for\s+.*?(?::|in).*?\n(?:[^\n]*\n)*?[^\n]*\.(?:create|insert)\(',
        'high',
        'Create in loop',
        'Individual inserts are slow',
        'Use bulk insert: bulk_create, insert_many, createMany'
    ),
    
    # Accessing relation without prefetch (Django specific)
    (
        r'for\s+(\w+)\s+in\s+.*?:\s*\n[^\n]*\1\.\w+_set\.',
        'high',
        'Django reverse relation in loop',
        'Accessing _set relation in loop causes N+1',
        'Use prefetch_related("relation_set") before loop'
    ),
    
    # Missing transaction for multiple writes
    (
        r'(?:\.save\(\)[^}]*\.save\(\)|\.create\([^}]*\.create\(|\.update\([^}]*\.update\()',
        'medium',
        'Multiple writes without transaction',
        'Multiple writes should be atomic',
        'Wrap in transaction/atomic block'
    ),

    # Distinct without index hint
    (
        r'\.distinct\(\s*\)(?!.*(?:order_by|index))',
        'low',
        'DISTINCT operation',
        'DISTINCT requires sorting/hashing; may be slow without proper index',
        'Ensure indexed column or review if DISTINCT is necessary'
    ),

    # GORM N+1 patterns (Go)
    (
        r'for\s+.*?range.*?\{[^}]*\.(?:Find|First|Take)\(',
        'high',
        'GORM N+1 (query in range loop)',
        'Running Find/First inside loop creates N+1 queries',
        'Use Preload() or batch with Where("id IN ?", ids)'
    ),
    (
        r'db\.(?:Find|First)\([^)]*\)(?!.*Preload)',
        'low',
        'GORM query without Preload',
        'Accessing associations later may trigger additional queries',
        'Add .Preload("Association") for eager loading'
    ),

    # ent N+1 patterns (Go)
    (
        r'for\s+.*?range.*?\{[^}]*\.Query\w+\(\)',
        'high',
        'ent N+1 (query in range loop)',
        'Running Query inside loop creates N+1 queries',
        'Use With*() eager loading or batch query'
    ),

    # sqlx/sqlc loop patterns (Go)
    (
        r'for\s+.*?range.*?\{[^}]*\.(?:Get|Select|Query)\w*\(',
        'high',
        'Go SQL N+1 (query in range loop)',
        'Running SQL query inside loop creates N+1 queries',
        'Use single query with WHERE IN clause or batch operations'
    ),
]

def analyze_file(filepath: Path) -> List[ORMIssue]:
    """Analyze a single file for ORM anti-patterns."""
    issues = []
    
    try:
        content = filepath.read_text(encoding='utf-8', errors='ignore')
    except Exception as e:
        return issues
    
    lines = content.split('\n')
    
    for pattern, severity, pattern_name, problem, solution in PATTERNS:
        # Search across the entire content for multi-line patterns
        for match in re.finditer(pattern, content, re.MULTILINE | re.DOTALL | re.IGNORECASE):
            # Find line number
            start_pos = match.start()
            line_num = content[:start_pos].count('\n') + 1
            
            # Get code snippet (the matched text, truncated)
            snippet = match.group(0)[:100].replace('\n', ' ').strip()
            if len(match.group(0)) > 100:
                snippet += '...'
            
            issues.append(ORMIssue(
                file=str(filepath),
                line=line_num,
                severity=severity,
                pattern=pattern_name,
                code_snippet=snippet,
                problem=problem,
                solution=solution
            ))
    
    return issues

def analyze_directory(dirpath: Path, extensions: List[str]) -> List[ORMIssue]:
    """Recursively analyze directory for ORM anti-patterns."""
    issues = []
    
    for ext in extensions:
        for filepath in dirpath.rglob(f'*{ext}'):
            # Skip common non-source directories
            if any(skip in str(filepath) for skip in ['node_modules', 'venv', '.git', '__pycache__', 'migrations']):
                continue
            issues.extend(analyze_file(filepath))
    
    return issues

def main():
    parser = argparse.ArgumentParser(description='Detect ORM anti-patterns')
    parser.add_argument('path', nargs='?', help='File or directory to analyze')
    parser.add_argument('--dir', '-d', help='Directory to analyze recursively')
    parser.add_argument('--ext', '-e', nargs='+', default=['.py', '.js', '.ts', '.rb', '.go'],
                        help='File extensions to analyze (default: .py .js .ts .rb .go)')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    target = args.dir or args.path
    
    if not target:
        # Read from stdin
        content = sys.stdin.read()
        if not content.strip():
            print("Error: No input provided", file=sys.stderr)
            sys.exit(1)
        
        # Create temp-like analysis
        issues = []
        for pattern, severity, pattern_name, problem, solution in PATTERNS:
            for match in re.finditer(pattern, content, re.MULTILINE | re.DOTALL | re.IGNORECASE):
                start_pos = match.start()
                line_num = content[:start_pos].count('\n') + 1
                snippet = match.group(0)[:100].replace('\n', ' ').strip()
                issues.append(ORMIssue(
                    file='stdin',
                    line=line_num,
                    severity=severity,
                    pattern=pattern_name,
                    code_snippet=snippet,
                    problem=problem,
                    solution=solution
                ))
        files_analyzed = 1
    else:
        path = Path(target)
        
        if path.is_file():
            issues = analyze_file(path)
            files_analyzed = 1
        elif path.is_dir():
            issues = analyze_directory(path, args.ext)
            files_analyzed = sum(1 for _ in path.rglob('*') if _.is_file() and _.suffix in args.ext)
        else:
            print(f"Error: '{target}' not found", file=sys.stderr)
            sys.exit(1)
    
    # Sort by severity
    severity_order = {'high': 0, 'medium': 1, 'low': 2}
    issues.sort(key=lambda x: (severity_order[x.severity], x.file, x.line))
    
    # Generate summary
    high = sum(1 for i in issues if i.severity == 'high')
    med = sum(1 for i in issues if i.severity == 'medium')
    
    if high > 0:
        summary = f"Found {high} high-severity ORM anti-patterns that likely cause performance issues."
    elif med > 0:
        summary = f"Found {med} medium-severity patterns to review."
    elif issues:
        summary = "Found minor optimization opportunities."
    else:
        summary = "No ORM anti-patterns detected."
    
    result = AnalysisResult(
        files_analyzed=files_analyzed,
        issues=[asdict(i) for i in issues],
        summary=summary
    )
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Files analyzed: {files_analyzed}")
        print(f"Summary: {summary}\n")
        
        if issues:
            print(f"Issues Found: {len(issues)}\n")
            
            current_file = None
            for issue in issues:
                if issue.file != current_file:
                    current_file = issue.file
                    print(f"📄 {current_file}")
                
                print(f"  Line {issue.line} [{issue.severity.upper()}] {issue.pattern}")
                print(f"    Code: {issue.code_snippet}")
                print(f"    Problem: {issue.problem}")
                print(f"    Fix: {issue.solution}")
                print()

if __name__ == "__main__":
    main()
