#!/usr/bin/env python3
"""
Parse and analyze EXPLAIN output from PostgreSQL, MySQL, or SQLite.
Usage: python parse_explain.py --pg explain_output.txt
       python parse_explain.py --mysql explain_output.txt
       python parse_explain.py --sqlite explain_output.txt
       echo "EXPLAIN output" | python parse_explain.py --pg
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict
from typing import List, Optional

@dataclass
class Warning:
    severity: str
    message: str
    suggestion: str

@dataclass
class ExplainAnalysis:
    database: str
    warnings: List[Warning]
    scan_types: List[str]
    estimated_rows: Optional[int]
    actual_rows: Optional[int]
    execution_time_ms: Optional[float]
    summary: str

def parse_postgresql(output: str) -> ExplainAnalysis:
    """Parse PostgreSQL EXPLAIN (ANALYZE) output."""
    warnings = []
    scan_types = []
    estimated_rows = None
    actual_rows = None
    execution_time = None
    
    lines = output.strip().split('\n')
    
    for line in lines:
        line_upper = line.upper()
        
        # Detect scan types
        if 'SEQ SCAN' in line_upper:
            scan_types.append('Sequential Scan')
            # Extract table name
            match = re.search(r'Seq Scan on (\w+)', line, re.IGNORECASE)
            table = match.group(1) if match else 'unknown'
            warnings.append(Warning(
                severity="high",
                message=f"Sequential scan on table '{table}'",
                suggestion=f"Consider adding index on filtered/joined columns of '{table}'"
            ))
        
        if 'INDEX SCAN' in line_upper or 'INDEX ONLY SCAN' in line_upper:
            scan_types.append('Index Scan' if 'ONLY' not in line_upper else 'Index Only Scan')
        
        if 'BITMAP' in line_upper:
            scan_types.append('Bitmap Scan')
        
        # Detect sort issues
        if 'SORT' in line_upper:
            if 'EXTERNAL' in line_upper or 'DISK' in line_upper:
                warnings.append(Warning(
                    severity="high",
                    message="Sort spilled to disk (external sort)",
                    suggestion="Increase work_mem or add index to avoid sort"
                ))
            scan_types.append('Sort')
        
        # Detect hash issues
        if 'HASH' in line_upper and 'BATCHES' in line_upper:
            batches_match = re.search(r'Batches:\s*(\d+)', line)
            if batches_match and int(batches_match.group(1)) > 1:
                warnings.append(Warning(
                    severity="medium",
                    message=f"Hash operation used {batches_match.group(1)} batches",
                    suggestion="Increase work_mem to fit hash table in memory"
                ))
        
        # Extract row estimates vs actuals
        rows_match = re.search(r'rows=(\d+).*?actual.*?rows=(\d+)', line, re.IGNORECASE)
        if rows_match:
            est = int(rows_match.group(1))
            act = int(rows_match.group(2))
            if estimated_rows is None:
                estimated_rows = est
                actual_rows = act
            # Check for significant estimation errors
            if est > 0 and (act / est > 10 or est / max(act, 1) > 10):
                warnings.append(Warning(
                    severity="medium",
                    message=f"Row estimate mismatch: estimated {est}, actual {act}",
                    suggestion="Run ANALYZE on table to update statistics"
                ))
        
        # Extract execution time
        time_match = re.search(r'Execution Time:\s*([\d.]+)\s*ms', line, re.IGNORECASE)
        if time_match:
            execution_time = float(time_match.group(1))
        
        # Detect nested loop with high rows
        if 'NESTED LOOP' in line_upper:
            scan_types.append('Nested Loop')
            rows_match = re.search(r'rows=(\d+)', line)
            if rows_match and int(rows_match.group(1)) > 10000:
                warnings.append(Warning(
                    severity="medium",
                    message="Nested loop with high row count",
                    suggestion="Consider if hash join would be more efficient"
                ))
    
    # Generate summary
    if not warnings:
        summary = "Query plan looks efficient."
    elif any(w.severity == "high" for w in warnings):
        summary = "Query has significant optimization opportunities."
    else:
        summary = "Query has minor optimization opportunities."
    
    return ExplainAnalysis(
        database="postgresql",
        warnings=[asdict(w) for w in warnings],
        scan_types=list(set(scan_types)),
        estimated_rows=estimated_rows,
        actual_rows=actual_rows,
        execution_time_ms=execution_time,
        summary=summary
    )

def parse_mysql(output: str) -> ExplainAnalysis:
    """Parse MySQL EXPLAIN output."""
    warnings = []
    scan_types = []
    estimated_rows = None
    execution_time = None
    
    lines = output.strip().split('\n')
    
    for line in lines:
        line_lower = line.lower()
        
        # Detect type column values
        if '\tall\t' in line_lower or '| all |' in line_lower:
            scan_types.append('Full Table Scan')
            warnings.append(Warning(
                severity="high",
                message="Full table scan detected (type=ALL)",
                suggestion="Add index on columns used in WHERE or JOIN"
            ))
        
        if '\tindex\t' in line_lower or '| index |' in line_lower:
            scan_types.append('Full Index Scan')
            warnings.append(Warning(
                severity="medium",
                message="Full index scan (type=index)",
                suggestion="Consider if a more selective index exists"
            ))
        
        if 'ref' in line_lower or 'eq_ref' in line_lower or 'const' in line_lower:
            scan_types.append('Index Lookup')
        
        if 'range' in line_lower:
            scan_types.append('Index Range Scan')
        
        # Detect extra column issues
        if 'using filesort' in line_lower:
            scan_types.append('Filesort')
            warnings.append(Warning(
                severity="medium",
                message="Using filesort for ORDER BY",
                suggestion="Add index that covers ORDER BY columns"
            ))
        
        if 'using temporary' in line_lower:
            scan_types.append('Temporary Table')
            warnings.append(Warning(
                severity="medium",
                message="Using temporary table",
                suggestion="Optimize GROUP BY/DISTINCT or add covering index"
            ))
        
        # Check for NULL key
        if re.search(r'\tNULL\t.*\tNULL\t', line) or '| NULL |' in line:
            warnings.append(Warning(
                severity="high",
                message="No index used (key=NULL)",
                suggestion="Add appropriate index for this query"
            ))
        
        # Extract rows estimate
        rows_match = re.search(r'\t(\d+)\t', line)
        if rows_match and estimated_rows is None:
            estimated_rows = int(rows_match.group(1))
    
    # Generate summary
    if not warnings:
        summary = "Query plan looks efficient."
    elif any(w.severity == "high" for w in warnings):
        summary = "Query has significant optimization opportunities."
    else:
        summary = "Query has minor optimization opportunities."
    
    return ExplainAnalysis(
        database="mysql",
        warnings=[asdict(w) for w in warnings],
        scan_types=list(set(scan_types)),
        estimated_rows=estimated_rows,
        actual_rows=None,
        execution_time_ms=execution_time,
        summary=summary
    )

def parse_sqlite(output: str) -> ExplainAnalysis:
    """Parse SQLite EXPLAIN QUERY PLAN output."""
    warnings = []
    scan_types = []

    lines = output.strip().split('\n')

    for line in lines:
        line_upper = line.upper()

        # SQLite EXPLAIN QUERY PLAN format: SCAN TABLE tablename
        if 'SCAN TABLE' in line_upper or ('SCAN' in line_upper and 'USING' not in line_upper):
            table_match = re.search(r'SCAN TABLE (\w+)', line, re.IGNORECASE)
            table = table_match.group(1) if table_match else 'unknown'
            scan_types.append('Full Table Scan')
            warnings.append(Warning(
                severity="high",
                message=f"Full table scan on '{table}'",
                suggestion=f"Add index on filtered columns of '{table}'"
            ))

        if 'SEARCH TABLE' in line_upper:
            scan_types.append('Index Search')

        if 'USING INDEX' in line_upper or 'USING COVERING INDEX' in line_upper:
            scan_types.append('Index Scan')

        if 'USING TEMP B-TREE' in line_upper:
            warnings.append(Warning(
                severity="medium",
                message="Using temporary B-tree for sorting",
                suggestion="Add index that matches ORDER BY to avoid sort"
            ))
            scan_types.append('Temp B-Tree Sort')

        if 'AUTOMATIC COVERING INDEX' in line_upper:
            warnings.append(Warning(
                severity="low",
                message="SQLite created automatic covering index",
                suggestion="Consider creating permanent index for frequently run queries"
            ))

    # Generate summary
    if not warnings:
        summary = "Query plan looks efficient."
    elif any(w.severity == "high" for w in warnings):
        summary = "Query has significant optimization opportunities."
    else:
        summary = "Query has minor optimization opportunities."

    return ExplainAnalysis(
        database="sqlite",
        warnings=[asdict(w) for w in warnings],
        scan_types=list(set(scan_types)),
        estimated_rows=None,
        actual_rows=None,
        execution_time_ms=None,
        summary=summary
    )

def main():
    parser = argparse.ArgumentParser(description='Parse EXPLAIN output')
    parser.add_argument('file', nargs='?', help='File containing EXPLAIN output')
    parser.add_argument('--pg', '--postgresql', action='store_true', help='Parse PostgreSQL EXPLAIN')
    parser.add_argument('--mysql', action='store_true', help='Parse MySQL EXPLAIN')
    parser.add_argument('--sqlite', action='store_true', help='Parse SQLite EXPLAIN QUERY PLAN')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()

    db_count = sum([args.pg, args.mysql, args.sqlite])
    if db_count == 0:
        print("Error: Specify --pg, --mysql, or --sqlite", file=sys.stderr)
        sys.exit(1)
    if db_count > 1:
        print("Error: Specify only one of --pg, --mysql, or --sqlite", file=sys.stderr)
        sys.exit(1)
    
    if args.file:
        try:
            with open(args.file) as f:
                output = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        output = sys.stdin.read()
    
    if not output.strip():
        print("Error: No EXPLAIN output provided", file=sys.stderr)
        sys.exit(1)
    
    if args.pg:
        result = parse_postgresql(output)
    elif args.mysql:
        result = parse_mysql(output)
    else:
        result = parse_sqlite(output)
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Database: {result.database.upper()}")
        print(f"Summary: {result.summary}")
        
        if result.scan_types:
            print(f"\nScan Types: {', '.join(result.scan_types)}")
        
        if result.estimated_rows:
            print(f"Estimated Rows: {result.estimated_rows:,}")
        if result.actual_rows:
            print(f"Actual Rows: {result.actual_rows:,}")
        if result.execution_time_ms:
            print(f"Execution Time: {result.execution_time_ms:.2f} ms")
        
        if result.warnings:
            print(f"\nWarnings: {len(result.warnings)}")
            for w in result.warnings:
                print(f"\n  [{w['severity'].upper()}] {w['message']}")
                print(f"    → {w['suggestion']}")

if __name__ == "__main__":
    main()
