#!/usr/bin/env python3
"""
Suggest indexes based on SQL query analysis.
Usage: python suggest_index.py "SELECT * FROM orders WHERE status = 'pending'"
       python suggest_index.py --file query.sql
       python suggest_index.py --file query.sql --table-info schema.json
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict
from typing import List, Dict, Optional, Set

@dataclass
class IndexSuggestion:
    table: str
    columns: List[str]
    type: str  # "btree", "covering", "partial", "composite"
    reason: str
    sql: str
    priority: str  # "high", "medium", "low"

def extract_tables_and_aliases(sql: str) -> Dict[str, str]:
    """Extract table names and their aliases."""
    tables = {}
    
    # FROM clause
    from_match = re.search(r'\bFROM\s+(\w+)(?:\s+(?:AS\s+)?(\w+))?', sql, re.IGNORECASE)
    if from_match:
        table = from_match.group(1)
        alias = from_match.group(2) or table
        tables[alias.lower()] = table.lower()
    
    # JOIN clauses
    join_matches = re.finditer(r'\bJOIN\s+(\w+)(?:\s+(?:AS\s+)?(\w+))?', sql, re.IGNORECASE)
    for match in join_matches:
        table = match.group(1)
        alias = match.group(2) or table
        tables[alias.lower()] = table.lower()
    
    return tables

def extract_where_columns(sql: str, tables: Dict[str, str]) -> Dict[str, List[tuple]]:
    """Extract columns used in WHERE clause with their operators."""
    table_columns: Dict[str, List[tuple]] = {}
    
    # Find WHERE clause
    where_match = re.search(r'\bWHERE\b(.+?)(?:\bGROUP BY\b|\bORDER BY\b|\bLIMIT\b|\bHAVING\b|$)', sql, re.IGNORECASE | re.DOTALL)
    if not where_match:
        return table_columns
    
    where_clause = where_match.group(1)
    
    # Pattern: table.column or just column with operator
    patterns = [
        (r'(\w+)\.(\w+)\s*(=|!=|<>|<|>|<=|>=|IN|LIKE|BETWEEN)', 'equality_or_range'),
        (r'(?<!\.)\b(\w+)\s*(=|!=|<>)\s*(?:\?|\'|\"|\d)', 'equality'),
        (r'(?<!\.)\b(\w+)\s*(IN)\s*\(', 'in_list'),
        (r'(?<!\.)\b(\w+)\s*(LIKE)\s*[\'"]', 'like'),
        (r'(?<!\.)\b(\w+)\s*(BETWEEN)', 'range'),
        (r'(?<!\.)\b(\w+)\s*(<|>|<=|>=)', 'range'),
    ]
    
    for pattern, op_type in patterns:
        for match in re.finditer(pattern, where_clause, re.IGNORECASE):
            if len(match.groups()) == 3:
                alias, column, operator = match.groups()
                table = tables.get(alias.lower(), alias.lower())
            else:
                column, operator = match.groups()
                # Default to first table if no alias
                table = list(tables.values())[0] if tables else 'unknown'
            
            if table not in table_columns:
                table_columns[table] = []
            table_columns[table].append((column.lower(), op_type, operator))
    
    return table_columns

def extract_join_columns(sql: str, tables: Dict[str, str]) -> Dict[str, List[str]]:
    """Extract columns used in JOIN conditions."""
    table_columns: Dict[str, List[str]] = {}
    
    # Pattern: ON table1.col1 = table2.col2
    join_matches = re.finditer(r'\bON\s+(\w+)\.(\w+)\s*=\s*(\w+)\.(\w+)', sql, re.IGNORECASE)
    
    for match in join_matches:
        alias1, col1, alias2, col2 = match.groups()
        
        table1 = tables.get(alias1.lower(), alias1.lower())
        table2 = tables.get(alias2.lower(), alias2.lower())
        
        if table1 not in table_columns:
            table_columns[table1] = []
        if table2 not in table_columns:
            table_columns[table2] = []
        
        table_columns[table1].append(col1.lower())
        table_columns[table2].append(col2.lower())
    
    return table_columns

def extract_order_columns(sql: str, tables: Dict[str, str]) -> Dict[str, List[str]]:
    """Extract columns used in ORDER BY."""
    table_columns: Dict[str, List[str]] = {}
    
    order_match = re.search(r'\bORDER BY\s+(.+?)(?:\bLIMIT\b|$)', sql, re.IGNORECASE)
    if not order_match:
        return table_columns
    
    order_clause = order_match.group(1)
    
    # Pattern: column or table.column
    for match in re.finditer(r'(\w+)(?:\.(\w+))?\s*(?:ASC|DESC)?', order_clause, re.IGNORECASE):
        if match.group(2):
            alias, column = match.groups()
            table = tables.get(alias.lower(), alias.lower())
        else:
            column = match.group(1)
            if column.upper() in ('ASC', 'DESC'):
                continue
            table = list(tables.values())[0] if tables else 'unknown'
        
        if table not in table_columns:
            table_columns[table] = []
        table_columns[table].append(column.lower())
    
    return table_columns

def extract_select_columns(sql: str, tables: Dict[str, str]) -> Dict[str, Set[str]]:
    """Extract columns in SELECT for covering index consideration."""
    table_columns: Dict[str, Set[str]] = {}
    
    select_match = re.search(r'\bSELECT\s+(.+?)\s+FROM\b', sql, re.IGNORECASE | re.DOTALL)
    if not select_match or '*' in select_match.group(1):
        return table_columns
    
    select_clause = select_match.group(1)
    
    for match in re.finditer(r'(\w+)(?:\.(\w+))?', select_clause, re.IGNORECASE):
        if match.group(2):
            alias, column = match.groups()
            table = tables.get(alias.lower(), alias.lower())
        else:
            column = match.group(1)
            if column.upper() in ('AS', 'DISTINCT', 'COUNT', 'SUM', 'AVG', 'MAX', 'MIN'):
                continue
            table = list(tables.values())[0] if tables else 'unknown'
        
        if table not in table_columns:
            table_columns[table] = set()
        table_columns[table].add(column.lower())
    
    return table_columns

def suggest_indexes(sql: str) -> List[IndexSuggestion]:
    """Analyze query and suggest indexes."""
    suggestions = []
    
    tables = extract_tables_and_aliases(sql)
    where_cols = extract_where_columns(sql, tables)
    join_cols = extract_join_columns(sql, tables)
    order_cols = extract_order_columns(sql, tables)
    select_cols = extract_select_columns(sql, tables)
    
    has_limit = bool(re.search(r'\bLIMIT\b', sql, re.IGNORECASE))
    
    for table in set(list(where_cols.keys()) + list(join_cols.keys())):
        index_columns = []
        reasons = []
        priority = "medium"
        
        # ESR rule: Equality, Sort, Range
        equality_cols = []
        range_cols = []
        
        if table in where_cols:
            for col, op_type, _ in where_cols[table]:
                if op_type in ('equality', 'in_list'):
                    equality_cols.append(col)
                elif op_type in ('range', 'like'):
                    range_cols.append(col)
                elif op_type == 'equality_or_range':
                    equality_cols.append(col)  # Treat as equality first
        
        # Add equality columns first
        for col in equality_cols:
            if col not in index_columns:
                index_columns.append(col)
                reasons.append(f"equality filter on {col}")
        
        # Add ORDER BY columns if LIMIT present
        if table in order_cols and has_limit:
            for col in order_cols[table]:
                if col not in index_columns:
                    index_columns.append(col)
                    reasons.append(f"ORDER BY {col}")
        
        # Add range columns last
        for col in range_cols:
            if col not in index_columns:
                index_columns.append(col)
                reasons.append(f"range filter on {col}")
        
        # Add join columns
        if table in join_cols:
            for col in join_cols[table]:
                if col not in index_columns:
                    index_columns.insert(0, col)  # Join cols often go first
                    reasons.append(f"JOIN condition on {col}")
                    priority = "high"
        
        if not index_columns:
            continue
        
        # Determine index type
        index_type = "composite" if len(index_columns) > 1 else "btree"
        
        # Check if covering index is possible
        if table in select_cols and len(select_cols[table]) <= 5:
            covering_cols = list(select_cols[table] - set(index_columns))
            if covering_cols and len(covering_cols) <= 3:
                index_columns.extend(covering_cols)
                index_type = "covering"
                reasons.append("includes SELECT columns for covering index")
        
        # Generate SQL
        col_list = ', '.join(index_columns)
        index_name = f"idx_{table}_{'_'.join(index_columns[:3])}"
        if len(index_columns) > 3:
            index_name += "_etc"
        
        sql_stmt = f"CREATE INDEX {index_name} ON {table}({col_list});"
        
        suggestions.append(IndexSuggestion(
            table=table,
            columns=index_columns,
            type=index_type,
            reason='; '.join(reasons),
            sql=sql_stmt,
            priority=priority
        ))
    
    # Sort by priority
    priority_order = {'high': 0, 'medium': 1, 'low': 2}
    suggestions.sort(key=lambda x: priority_order[x.priority])
    
    return suggestions

def main():
    parser = argparse.ArgumentParser(description='Suggest indexes for SQL query')
    parser.add_argument('query', nargs='?', help='SQL query string')
    parser.add_argument('--file', '-f', help='Read query from file')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if args.file:
        try:
            with open(args.file) as f:
                sql = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    elif args.query:
        sql = args.query
    else:
        sql = sys.stdin.read()
    
    if not sql.strip():
        print("Error: No query provided", file=sys.stderr)
        sys.exit(1)
    
    suggestions = suggest_indexes(sql)
    
    if args.json:
        print(json.dumps([asdict(s) for s in suggestions], indent=2))
    else:
        if not suggestions:
            print("No index suggestions for this query.")
            return
        
        print(f"Index Suggestions: {len(suggestions)}\n")
        for i, s in enumerate(suggestions, 1):
            print(f"{i}. [{s.priority.upper()}] {s.table}")
            print(f"   Columns: {', '.join(s.columns)}")
            print(f"   Type: {s.type}")
            print(f"   Reason: {s.reason}")
            print(f"   SQL: {s.sql}")
            print()

if __name__ == "__main__":
    main()
