# Execution Plan Analysis

## Contents
- PostgreSQL EXPLAIN (running, key metrics, reading output, red flags, fixes)
- MySQL EXPLAIN (running, key columns, type ranking, extra field)
- MongoDB explain() (running, key fields, efficiency ratios, stage types)
- Common Optimization Actions

## PostgreSQL EXPLAIN

### Running EXPLAIN
```sql
EXPLAIN (ANALYZE, BUFFERS, FORMAT TEXT) SELECT ...;
```

### Key Metrics
| Metric | Meaning | Target |
|--------|---------|--------|
| Seq Scan | Full table scan | Avoid on large tables |
| Index Scan | Uses index, fetches rows | Good for selective queries |
| Index Only Scan | Covered query | Best for read performance |
| Bitmap Scan | Multiple index conditions | OK for medium selectivity |
| Nested Loop | Row-by-row join | Good for small inner table |
| Hash Join | Builds hash table | Good for large unsorted joins |
| Merge Join | Sorted merge | Good for pre-sorted data |
| Sort | Sorting operation | Check if disk sort (external) |

### Reading Output
```
Seq Scan on orders  (cost=0.00..1234.00 rows=50000 width=64)
                     ↑          ↑        ↑          ↑
                 startup     total   estimated   row width
                  cost       cost     rows       (bytes)
```

### Red Flags
- `rows=1` but actual returns 10000+ (statistics outdated)
- `Seq Scan` on large tables with WHERE clause
- `Sort Method: external merge Disk` (memory exceeded)
- `Buffers: read` >> `Buffers: hit` (cache misses)
- Nested loop with large inner table

### Fixes
```sql
-- Update statistics
ANALYZE table_name;

-- Add missing index
CREATE INDEX CONCURRENTLY idx_name ON table(column);

-- Increase work_mem for sorts
SET work_mem = '256MB';
```

## MySQL EXPLAIN

### Running EXPLAIN
```sql
EXPLAIN ANALYZE SELECT ...;  -- MySQL 8.0+
EXPLAIN FORMAT=JSON SELECT ...;
```

### Key Columns
| Column | Good Values | Bad Values |
|--------|-------------|------------|
| type | const, eq_ref, ref, range | ALL, index |
| key | Index name | NULL |
| key_len | Full index length | Partial use |
| rows | Low estimate | High (millions) |
| Extra | Using index | Using filesort, Using temporary |

### Type Ranking (best to worst)
1. `system` / `const` - Single row
2. `eq_ref` - Unique index lookup
3. `ref` - Non-unique index lookup
4. `range` - Index range scan
5. `index` - Full index scan
6. `ALL` - Full table scan ❌

### Extra Field Red Flags
- `Using filesort` - Needs optimization
- `Using temporary` - Temp table created
- `Using where` - Post-index filtering (may be OK)
- `Using join buffer` - No index for join

### Key Length Interpretation
```sql
-- Index on (a INT, b VARCHAR(50), c DATE)
-- Max key_len = 4 + 52*3 + 3 = 163 (utf8)
-- If key_len = 4, only 'a' used
-- If key_len = 160, 'a' and 'b' used
```

## MongoDB explain()

### Running Explain
```javascript
db.collection.find({...}).explain("executionStats")
```

### Key Fields
```javascript
{
  "queryPlanner": {
    "winningPlan": {
      "stage": "FETCH",  // or IXSCAN, COLLSCAN
      "inputStage": {
        "stage": "IXSCAN",
        "indexName": "status_1_created_at_-1"
      }
    }
  },
  "executionStats": {
    "nReturned": 100,          // Documents returned
    "totalKeysExamined": 150,  // Index entries scanned
    "totalDocsExamined": 150,  // Documents scanned
    "executionTimeMillis": 5
  }
}
```

### Efficiency Ratios
```
# Good: nReturned ≈ totalDocsExamined
# Bad: totalDocsExamined >> nReturned (over-scanning)

# Ideal covered query:
nReturned = totalKeysExamined
totalDocsExamined = 0
```

### Stage Types
| Stage | Description | Concern |
|-------|-------------|---------|
| COLLSCAN | Collection scan | ❌ Add index |
| IXSCAN | Index scan | ✓ Good |
| FETCH | Get documents | Normal |
| SORT | In-memory sort | Watch memory |
| SORT_KEY_GENERATOR | Sort preparation | Check index |
| PROJECTION_COVERED | From index only | ✓ Best |

### Red Flags
- `COLLSCAN` on large collections
- `totalDocsExamined` >> `nReturned`
- `hasSortStage: true` without supporting index
- `rejectedPlans` contains better-looking plan

## Common Optimization Actions

### After EXPLAIN Analysis
1. **Add index** for Seq Scan / COLLSCAN
2. **Extend compound index** for partial usage
3. **Reorder index columns** (ESR rule for MongoDB)
4. **Update statistics** (ANALYZE in PostgreSQL)
5. **Increase memory** for sort/hash operations
6. **Rewrite query** if plan shows inefficient path
7. **Add covering columns** to avoid FETCH
